import _memoize from 'lru-memoize';

import { debounceFunc } from '../helpers';

/**
 * Привязывает контекст метода класса к экземпляру класса
 *
 * Аналог `this.method = this.method.bind(this)` в конструкторе, просто более декларативный.
 * Используется в основном для обработчиков событий в JSX
 *
 * Взято отсюда: https://github.com/typed-decorators/autobind/blob/master/src/autobind.ts (лицензия ISC)
 */
// tslint:disable-next-line:ban-types
export function autobind<F extends Function>(
   _target: any,
   name: string,
   descriptor: TypedPropertyDescriptor<F>,
): TypedPropertyDescriptor<F> {
   const { enumerable, configurable, value } = descriptor;

   const boundMethod = Symbol(`autobind/${name}`);

   return {
      configurable,
      enumerable,

      get(this: { [boundMethod]: any }) {
         if (!this[boundMethod]) {
            this[boundMethod] = value!.bind(this);
         }

         return this[boundMethod];
      },

      // tslint:disable-next-line:no-shadowed-variable
      set(v: F) {
         Object.defineProperty(this, name, {
            configurable: true,
            enumerable: true,
            value: v,
            writable: true,
         });
      },
   };
}

// tslint:disable-next-line:ban-types
export function bindAndMemoize<F extends Function>(
   _target: any,
   name: string,
   descriptor: TypedPropertyDescriptor<F>,
): TypedPropertyDescriptor<F> {
   const { enumerable, configurable, value } = descriptor;

   const memoizedMethod = Symbol(`memoize/${name}`);

   return {
      configurable,
      enumerable,

      get(this: { [memoizedMethod]: any }) {
         if (!this[memoizedMethod]) {
            // 100 штук должно хватить, вряд ли будут списки больше
            this[memoizedMethod] = _memoize(100)(value!.bind(this));
         }

         return this[memoizedMethod];
      },

      // tslint:disable-next-line:no-shadowed-variable
      set(v: F) {
         Object.defineProperty(this, name, {
            configurable: true,
            enumerable: true,
            value: v,
            writable: true,
         });
      },
   };
}

export function debounce(ms: number) {
   // tslint:disable-next-line:ban-types only-arrow-functions
   return <F extends Function>(
      _target: any,
      name: string,
      descriptor: TypedPropertyDescriptor<F>,
   ): TypedPropertyDescriptor<F> => {
      const { enumerable, configurable, value } = descriptor;

      const debouncedMethod = Symbol(`debounced/${name}`);

      return {
         configurable,
         enumerable,

         get(this: { [debouncedMethod]: any }) {
            if (!this[debouncedMethod]) {
               this[debouncedMethod] = debounceFunc(value!.bind(this), ms);
            }

            return this[debouncedMethod];
         },

         // tslint:disable-next-line:no-shadowed-variable
         set(v: F) {
            Object.defineProperty(this, name, {
               configurable: true,
               enumerable: true,
               value: v,
               writable: true,
            });
         },
      };
   };
}
