import { array, number, object, string } from 'yup';

import { IValidationResult, record, validateEntity } from './validation';

describe('utils|validation', () => {
   describe('validateEntity', () => {
      interface Entity {
         x?: number;
      }

      const schema = object<Entity>({
         x: number().max(10).required(),
      });

      it('should be valid', () => {
         const r = validateEntity(schema, ['x'], { x: 2 }, null);

         expect(r).toEqual({
            errors: new Map(),
            isValid: true,
         } as IValidationResult<Entity>);
      });

      it('should be invalid', () => {
         const r = validateEntity(schema, ['x'], { x: 20 }, null);

         expect(r).toEqual({
            errors: new Map([['x', ['x must be less than or equal to 10']]]),
            isValid: false,
         } as IValidationResult<Entity>);
      });
   });

   describe('record', () => {
      const internalSchema = object({
         name: string().oneOf(['x', 'y']).required(),
         age: number().required(),
      });

      it('should work for array.of()', () => {
         const schema = array().of(internalSchema);

         expect(() => schema.validateSync([{ name: 'a', age: 0 }])).toThrow(
            '[0].name must be one of the following values: x, y',
         );
      });

      it('should work for record.of()', () => {
         const schema: any = record(internalSchema);

         expect(() => schema.validateSync({ x: { name: 'a', age: 0 } })).toThrow(
            'x.name must be one of the following values: x, y',
         );
      });
   });
});
