import { array, lazy, object, ObjectSchemaDefinition, Schema, ValidationError } from 'yup';

export interface IValidationResult<T> {
   errors: Map<keyof T, string[]>;
   isValid: boolean;
}

export type Validator<T> = (v: T, context?: any) => IValidationResult<T>;

export function setAsArray() {
   return array().transform((value: Set<string>, originalValue: Set<string>) => Array.from(originalValue));
}

export function getValidResult<T>() {
   return {
      errors: new Map(),
      isValid: true,
   } as IValidationResult<T>;
}

export function validateEntity<T>(
   schema: Schema<T>,
   fields: (keyof T)[],
   value: T,
   context: any,
): IValidationResult<T> {
   const result = getValidResult<T>();

   for (const field of fields) {
      try {
         schema.validateSyncAt(field as string, value, { context });
      } catch (e) {
         if (!(e instanceof ValidationError)) {
            throw e;
         }

         result.errors.set(field, e.errors);
         result.isValid = false;
      }
   }

   return result;
}

/**
 * Создаёт функцию для валидации объекта по схеме
 */
export function createValidator<T extends object>(shape: ObjectSchemaDefinition<T>): Validator<T> {
   const schema = object().shape(shape);
   const fields = Object.keys(shape) as (keyof T)[];

   return (params: T, context = {}) => validateEntity(schema, fields, params, context);
}

export function record<T extends object>(schema: Schema<T>): Schema<Record<string, T>> {
   return lazy(
      (v: T): Schema<any> =>
         object(
            Object.keys(v).reduce((acc, key) => {
               acc[key] = schema;

               return acc;
            }, {}),
         ),
   );
}
