/* tslint:disable:object-literal-sort-keys */
const path = require('path');

const BundleAnalyzerPlugin = require('webpack-bundle-analyzer').BundleAnalyzerPlugin;
const CleanWebpackPlugin = require('clean-webpack-plugin').CleanWebpackPlugin;
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const nodeExternals = require('webpack-node-externals');
const { CheckerPlugin } = require('awesome-typescript-loader');
const getCSSModuleLocalIdent = require('react-dev-utils/getCSSModuleLocalIdent');

const cssRegex = /\.css$/;
const cssModuleRegex = /\.module\.css$/;

const shouldUseSourceMap = process.env.GENERATE_SOURCEMAP !== 'false';

const paths = {
   servedPath: '.',
};

module.exports = function ({ production: isEnvProduction }) {
   const isEnvDevelopment = !isEnvProduction;
   console.log('production:', isEnvProduction);

   const publicPath = isEnvProduction ? paths.servedPath : isEnvDevelopment && '/';

   const shouldUseRelativeAssetPaths = publicPath === './';
   const getStyleLoaders = (cssOptions, preProcessor) => {
      const loaders = [
         isEnvDevelopment && require.resolve('style-loader'),
         isEnvProduction && {
            loader: MiniCssExtractPlugin.loader,
            options: Object.assign({}, shouldUseRelativeAssetPaths ? { publicPath: '.' } : undefined),
         },
         {
            loader: require.resolve('css-loader'),
            options: cssOptions,
         },
         {
            loader: require.resolve('postcss-loader'),
            options: {
               ident: 'postcss',
               plugins: () => [
                  require('postcss-flexbugs-fixes'),
                  require('postcss-preset-env')({
                     autoprefixer: { flexbox: 'no-2009' },
                     stage: 3,
                  }),
               ],
               sourceMap: isEnvProduction && shouldUseSourceMap,
            },
         },
      ].filter(Boolean);
      if (preProcessor) {
         loaders.push({
            loader: require.resolve(preProcessor),
            options: {
               sourceMap: isEnvProduction && shouldUseSourceMap,
            },
         });
      }
      return loaders;
   };

   return {
      bail: isEnvProduction,
      devtool: isEnvProduction
         ? shouldUseSourceMap
            ? 'source-map'
            : false
         : isEnvDevelopment && 'cheap-module-source-map',
      mode: isEnvDevelopment ? 'development' : 'production',
      entry: {
         index: './src/library.ts',
      },
      module: {
         rules: [
            {
               test: /\.tsx?$/,
               exclude: /\.(test|stories)\.tsx?$/,
               loader: 'awesome-typescript-loader',
               options: {
                  jsx: 'react',
               },
            },
            {
               test: cssRegex,
               exclude: cssModuleRegex,
               use: getStyleLoaders({
                  importLoaders: 1,
                  sourceMap: isEnvProduction && shouldUseSourceMap,
               }),
               sideEffects: true,
            },
            {
               test: cssModuleRegex,
               use: getStyleLoaders({
                  importLoaders: 1,
                  sourceMap: isEnvProduction && shouldUseSourceMap,
                  modules: {
                     getLocalIdent: getCSSModuleLocalIdent,
                  },
               }),
            },
            {
               test: /\.svg$/,
               use: ['@svgr/webpack', 'url-loader'],
            },
            {
               test: /\.png$/,
               loader: 'url-loader',
            },
            {
               test: /\.(woff2)$/,
               loader: 'file-loader',
               options: {
                  name: 'assets/[name].[hash:6].[ext]',
               },
            },
         ],
      },
      optimization: { minimize: false },
      output: {
         filename: '[name].js',
         path: path.resolve(__dirname, 'package', 'dist'),
         library: 'icui',
         libraryTarget: 'umd',
      },
      plugins: [
         new CleanWebpackPlugin(),
         new CheckerPlugin(),
         new MiniCssExtractPlugin({ filename: '[name].css' }),
         // new BundleAnalyzerPlugin({
         //    analyzerMode: 'static',
         //    generateStatsFile: true,
         //    defaultSizes: 'gzip',
         //    openAnalyzer: isEnvDevelopment,
         // }),
      ].filter(Boolean),
      resolve: {
         extensions: ['.ts', '.tsx', '.js', '.jsx'],
      },
      externals: [nodeExternals()],
      target: 'web',
   };
};
