import React, { forwardRef, useEffect, useImperativeHandle, useRef, useState } from 'react';
import { useTheme } from '@yandex-cloud/uikit';
import { classNames } from '@yandex-infracloud-ui/libs';
import { IDisposable, Uri } from 'monaco-editor';
import { editor as monacoEditor } from 'monaco-editor/esm/vs/editor/editor.api';

import {
    EditorRef,
    getChangeValidator,
    getDefaultEditorOptions,
    getSizeUpdater,
    MAX_ROWS,
    MIN_ROWS,
    MonacoEditorProps,
    PlaceholderWidget,
} from '../../models';

import { defineInfracloudMonacoThemesSync } from '../../services';

import classes from './MonacoEditor.module.css';

// регистрация тем по умолчанию
defineInfracloudMonacoThemesSync(monacoEditor);

export const MonacoEditor = forwardRef<EditorRef, MonacoEditorProps>(
    (
        {
            className,
            rows,
            maxRows = MAX_ROWS,
            minRows = MIN_ROWS,
            isAutoRows = true,
            isAutoMaxHeight = false,
            value,
            onUpdate,
            readonly,
            uri: schemaUri,
            options,
            placeholder,
            onChangeValidate,
            theme,
            onSetup,
            showBorder = true,
        },
        ref,
    ) => {
        const [commonTheme] = useTheme();

        const [focused, setFocused] = useState(false);
        const lineHeightRef = useRef(options?.lineHeight ?? 13);
        const initialValueRef = useRef(value);
        const editorDivRef = useRef<HTMLDivElement>(null);
        const editorRef = useRef<monacoEditor.IStandaloneCodeEditor>();

        // Инициализация редактора и навешивания события
        useEffect(() => {
            if (!editorDivRef.current) {
                return () => undefined;
            }
            const editorElement = editorDivRef.current;
            const model = monacoEditor.createModel(
                initialValueRef.current,
                options?.language,
                schemaUri ? Uri.parse(schemaUri).with({ query: `timestamp=${Date.now()}` }) : undefined,
            );

            const editorOptions: monacoEditor.IStandaloneEditorConstructionOptions = {
                ...getDefaultEditorOptions(),
                readOnly: readonly || !onUpdate,
                theme: theme ?? (commonTheme === 'light' ? 'infracloud-ui' : 'infracloud-ui-dark'),
                model,
                ...options,
            };

            const editor = monacoEditor.create(editorElement, editorOptions);

            const lineHeight = editor.getOption(monacoEditor.EditorOption.lineHeight);
            lineHeightRef.current = lineHeight;

            // Отслеживание высоты (подстройка по контент).
            const updateSize = getSizeUpdater({
                monacoEditorLayoutProps: { isAutoRows, rows, maxRows, minRows, isAutoMaxHeight },
                editor,
                editorElement,
                lineHeight,
            });

            window.addEventListener('resize', updateSize, false);
            window.setTimeout(updateSize);

            editorRef.current = editor;

            const disposables: IDisposable[] = [
                editor,
                editor.onDidFocusEditorWidget(() => setFocused(true)),
                editor.onDidBlurEditorWidget(() => setFocused(false)),
            ];

            if (placeholder) {
                disposables.push(new PlaceholderWidget(placeholder, editor));
            }

            if (onUpdate && !readonly) {
                disposables.push(
                    editor.onDidChangeModelContent(() => {
                        onUpdate(editor.getModel()?.getValue() ?? '');
                        updateSize();
                    }),
                );
            }

            disposables.push(getChangeValidator({ editor, editorApi: monacoEditor, onChangeValidate }));

            if (onSetup) {
                disposables.push(onSetup(editor));
            }

            return () => {
                window.removeEventListener('resize', updateSize);
                editorRef.current = undefined;
                disposables.forEach(d => d.dispose());
            };
        }, [commonTheme, maxRows, minRows, onChangeValidate, onUpdate, onSetup, rows, schemaUri, isAutoRows, options]);

        // Обновление внутреннего значения при изменении внешнего
        useEffect(() => {
            const editor = editorRef.current;
            if (editor) {
                const editorValue = editor.getValue();
                if (value !== editorValue) {
                    editor.setValue(value);
                }
            }
        }, [value]);

        // Возможность из-вне обратиться к инстансу редактора (для продвинутых нужд)
        useImperativeHandle(ref, () => ({
            getRawEditor() {
                return editorRef.current;
            },
            getMonacoEditorApi() {
                return monacoEditor;
            },
        }));

        const wrapperClassName = classNames(className, {
            [classes.focused]: showBorder && focused,
            [classes.wrapper]: showBorder,
        });

        return (
            <div className={wrapperClassName}>
                <div
                    className={classes.editor}
                    ref={editorDivRef}
                    style={{ minHeight: minRows * lineHeightRef.current }}
                />
            </div>
        );
    },
);

MonacoEditor.displayName = 'MonacoEditor';

// default export is for lazy-loading (code splitting)
export default MonacoEditor;
