import React, { useCallback, useEffect, useRef, useState } from 'react';
import type { editor as monacoEditor, IDisposable } from 'monaco-editor/esm/vs/editor/editor.api';
import { classNames, Skeleton } from '@yandex-infracloud-ui/libs';

// noinspection ES6PreferShortImport
import { LazyMonacoEditor } from '../MonacoEditor/LazyMonacoEditor';

import { registerYamlLanguage } from './registerYamlLanguage';
import { getYamlEditorOptions } from './yamlEditorOptions';
import type { MonacoEditorProps } from '../../models';

import classes from './YamlEditor.module.css';

// настройка саджеста по Enter
const setupCompletion = (editor: monacoEditor.IStandaloneCodeEditor): IDisposable => {
    return editor.onDidChangeModelContent(changeEvent => {
        const { changes } = changeEvent;
        if (changes.length === 1) {
            const { text } = changes[0];
            const newLinePattern = /^\n( )*$/.test(text);
            if (newLinePattern) {
                window.setTimeout(() => editor.trigger('yaml', 'editor.action.triggerSuggest', null), 0);
            }
        }
    });
};

export type YamlEditorProps = MonacoEditorProps;

export const YamlEditor: React.FC<YamlEditorProps> = React.memo((props: YamlEditorProps) => {
    const { className, options, onSetup: onSetupCustom } = props;
    const [initiated, setInitiated] = useState(false);
    const optionsRef = useRef<monacoEditor.IStandaloneEditorConstructionOptions>({});

    useEffect(() => {
        let dispose: (() => void) | undefined;

        (async function queryProviderInit() {
            const result = await registerYamlLanguage();
            const { language } = result;
            dispose = result.dispose;

            optionsRef.current = {
                ...getYamlEditorOptions(),
                language,
                ...options,
            };

            setInitiated(true);
        })();

        return dispose;
    }, [options]);

    const onSetup = useCallback((editor: monacoEditor.IStandaloneCodeEditor): IDisposable => {
        const disposable = [setupCompletion(editor)];

        if (onSetupCustom) {
            disposable.push(onSetupCustom(editor));
        }

        return {
            dispose() {
                console.log(disposable);
                disposable.filter(Boolean).forEach(e => e.dispose());
            },
        };
    }, []);

    return initiated ? (
        <>
            <LazyMonacoEditor
                {...props}
                className={classNames(classes.wrapper, className)}
                options={optionsRef.current}
                onSetup={onSetup}
            />
        </>
    ) : (
        <Skeleton items={[{ size: 44, space: 0, width: '100%' }]} />
    );
});

YamlEditor.displayName = 'YamlEditor';
