import type { IDisposable } from 'monaco-editor';
import { getSetDifference } from '@yandex-infracloud-ui/libs';
import type { SchemasSettings } from 'monaco-yaml';

import { memoAsyncEffect } from '../../utils';
import { importMonacoApi, importMonacoYaml } from '../../services';

const languageId = 'yaml';

interface RegisterLanguageResult extends IDisposable {
    language: string;
}

const setYamlInitialOptions = memoAsyncEffect(async () => {
    await importMonacoYaml();
    const monaco = await importMonacoApi();
    monaco.languages.yaml.yamlDefaults.setDiagnosticsOptions({
        schemas: [],
        enableSchemaRequest: true,
        hover: true,
        hoverSettings: {
            showSource: false,
            showTitle: false,
        },
        completion: true,
        validate: true,
    });
});

export async function registerYamlLanguageImpl(): Promise<RegisterLanguageResult> {
    await setYamlInitialOptions();

    return {
        dispose() {
            return undefined;
        },
        language: languageId,
    };
}

export const registerYamlLanguage = memoAsyncEffect(registerYamlLanguageImpl);

const schemaMap: Map<string, SchemasSettings> = new Map();

export type SchemaSettingsItem = {
    schemaSettings: SchemasSettings;
    replace?: boolean;
};

export async function registerYamlJSONSchema(...schemaSettingsList: SchemaSettingsItem[]): Promise<void> {
    await setYamlInitialOptions();
    const monaco = await importMonacoApi();
    const uriMap = new Map(schemaSettingsList.map(item => [item.schemaSettings.uri, item]));
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    const getSchema = (uri: string) => uriMap.get(uri)!.schemaSettings;
    const { added, unchanged } = getSetDifference(new Set(schemaMap.keys()), new Set(uriMap.keys()));
    const schemas = monaco.languages.yaml.yamlDefaults.diagnosticsOptions.schemas;
    if (!schemas) {
        return;
    }

    if (added.size > 0) {
        schemas.push(...Array.from(added).map(getSchema));
    }

    if (unchanged.size > 0) {
        const replacedUriSet = new Set(Array.from(unchanged).filter(uri => uriMap.get(uri)!.replace));
        if (replacedUriSet.size > 0) {
            for (const oldSchema of schemas) {
                if (replacedUriSet.has(oldSchema.uri)) {
                    const newSchema = getSchema(oldSchema.uri);
                    oldSchema.fileMatch = newSchema.fileMatch;
                    oldSchema.schema = newSchema.schema;
                }
            }
        }
    }
}
