import { RefObject, useCallback, useEffect, useRef, useState } from 'react';

import { TIMES_IN_MS, UpdateTimerMode, useUpdateTimer } from '@yandex-infracloud-ui/libs';
import type { editor } from 'monaco-editor/esm/vs/editor/editor.api';

import { EditorRef, getChangeValidator, MonacoEditorValidationInfo } from '../models';

export function useStatusDataByEditorRef({ editorRef }: { editorRef?: RefObject<EditorRef> }): {
    validationInfo?: MonacoEditorValidationInfo | null;
} {
    const [currentEditorRef, setEditorRef] = useState<editor.IStandaloneCodeEditor | null>(null);
    const editorApi = useRef<typeof editor | null>(null);

    const updateRef = useCallback(() => {
        const ref = editorRef?.current;
        const rawEditor = ref?.getRawEditor?.() ?? null;

        if (ref && rawEditor && !editorApi.current) {
            editorApi.current = ref.getMonacoEditorApi();
        }

        if (rawEditor !== currentEditorRef) {
            setEditorRef(rawEditor);
        }
    }, [currentEditorRef]);

    useUpdateTimer({
        callback: updateRef,
        fast: TIMES_IN_MS.Second,
        slow: TIMES_IN_MS.Second,
        mode: UpdateTimerMode.Fast,
    });

    const [currentValidationInfo, setValidationInfo] = useState<MonacoEditorValidationInfo | null>(null);

    const onChangeValidate = useCallback(({ validationInfo }) => setValidationInfo(validationInfo), []);

    useEffect(() => {
        if (!currentEditorRef || !editorApi.current) {
            return undefined;
        }

        const validationDisposable = getChangeValidator({
            editor: currentEditorRef,
            editorApi: editorApi.current,
            onChangeValidate,
        });

        return () => {
            validationDisposable.dispose();
        };
    }, [currentEditorRef]);

    return {
        validationInfo: currentValidationInfo,
    };
}
