import type { editor as monacoEditor, IDisposable, Uri } from 'monaco-editor/esm/vs/editor/editor.api';
import { getMessageIndexesByStatus, getMonacoEditorMessage, getMonacoEditorMessageSummary } from './message';
import { MAX_ROWS, MIN_ROWS, MonacoEditorLayoutProps, MonacoEditorProps } from './monaco';

export function getSizeUpdater({
    monacoEditorLayoutProps,
    editor,
    editorElement,
    lineHeight,
}: {
    monacoEditorLayoutProps: MonacoEditorLayoutProps;
    editor: monacoEditor.IStandaloneCodeEditor;
    editorElement: HTMLDivElement;
    lineHeight: number;
}): () => void {
    const { isAutoRows, maxRows = MAX_ROWS, minRows = MIN_ROWS, rows, isAutoMaxHeight } = monacoEditorLayoutProps;

    let prevHeight = 0;
    let prevWidth = 0;

    const getCurrentRows = () => editor.getModel()?.getValue().split('\n').length ?? 1;
    const getCurrentFullHeight = () => getCurrentRows() * lineHeight;

    const getFixedHeight = () => {
        let targetRows = minRows;
        if (rows) {
            targetRows = rows;
        } else {
            const currentRows = getCurrentRows();
            targetRows = Math.min(Math.max(minRows, currentRows), maxRows);
        }
        const height = targetRows * lineHeight;

        return height;
    };

    const updateHeight = () => {
        const { height: availableHeight, width } = editorElement.getBoundingClientRect();
        let height = availableHeight;
        if (isAutoRows) {
            if (!isAutoMaxHeight) {
                height = Math.min(getCurrentFullHeight(), availableHeight);
            }
        } else {
            height = getFixedHeight();
        }

        const heightChanged = prevHeight !== height;
        const widthChanged = width !== prevWidth;

        if (heightChanged || widthChanged) {
            prevHeight = height;
            prevWidth = width;
            editor.layout({ height, width });
        }
    };

    return updateHeight;
}

export function getChangeValidator({
    editor,
    editorApi,
    onChangeValidate,
}: {
    editor: monacoEditor.IStandaloneCodeEditor;
    editorApi: typeof monacoEditor;
    onChangeValidate: MonacoEditorProps['onChangeValidate'];
}): IDisposable {
    if (!onChangeValidate) {
        return {
            dispose: () => undefined,
        };
    }

    const runValidate = (uri: Uri) => {
        const markers = editorApi.getModelMarkers({ resource: uri });
        const messages = markers.map(marker => getMonacoEditorMessage({ marker }));
        const summary = getMonacoEditorMessageSummary({ messages });
        const messageIndexesByStatus = getMessageIndexesByStatus({ messages });
        onChangeValidate({
            validationInfo: {
                rawMarkers: markers,
                messages,
                summary,
                messageIndexesByStatus,
            },
        });
    };

    const getUri = () => editor.getModel()?.uri;

    const firstModelUri = getUri();
    if (firstModelUri) {
        runValidate(firstModelUri);
    }

    return editorApi.onDidChangeMarkers(uriList => {
        const modelUri = getUri()?.toString();
        if (!modelUri) {
            return;
        }
        const currentUri = uriList.find(uri => uri.toString() === modelUri);
        if (!currentUri) {
            return;
        }
        runValidate(currentUri);
    });
}
