import type { editor } from 'monaco-editor';
import { MarkerSeverity } from 'monaco-editor';

export type MonacoEditorMessageStatus = 'error' | 'warning' | 'info' | 'none';

export const monacoEditorMessageStatusBySeverity: Record<MarkerSeverity, MonacoEditorMessageStatus> = {
    [MarkerSeverity.Error]: 'error',
    [MarkerSeverity.Warning]: 'warning',
    [MarkerSeverity.Info]: 'info',
    [MarkerSeverity.Hint]: 'none',
};

export const monacoEditorMessageStatusOrder: MonacoEditorMessageStatus[] = ['error', 'warning', 'info', 'none'];
export const monacoEditorMessageStatusOrderIndexes: Record<MonacoEditorMessageStatus, number> = Object.fromEntries(
    monacoEditorMessageStatusOrder.map((status, i) => [status, i]),
) as Record<MonacoEditorMessageStatus, number>;

export interface MonacoEditorPosition {
    line: number;
    column: number;
}

export interface MonacoEditorRange {
    start: MonacoEditorPosition;
    end: MonacoEditorPosition;
}

export interface MonacoEditorMessage {
    status: MonacoEditorMessageStatus;
    message: string;
    range: MonacoEditorRange;
    source: string;
}

export type MonacoEditorMessageCountByStatus = Record<MonacoEditorMessageStatus, number>;

export interface MonacoEditorMessageSummary {
    summaryStatus: MonacoEditorMessageStatus;
    countByStatus: MonacoEditorMessageCountByStatus;
    totalCount: number;
}

export type MessageIndexesByStatus = Record<MonacoEditorMessageStatus, Set<number>>;

export interface MonacoEditorValidationInfo {
    rawMarkers: editor.IMarker[];
    summary: MonacoEditorMessageSummary;
    messages: MonacoEditorMessage[];
    messageIndexesByStatus: MessageIndexesByStatus;
}

export function getMonacoEditorMessage({ marker }: { marker: editor.IMarker }): MonacoEditorMessage {
    const { message, severity, source, startColumn, endColumn, startLineNumber, endLineNumber } = marker;
    const editorMessage: MonacoEditorMessage = {
        message,
        source: source ?? '',
        status: monacoEditorMessageStatusBySeverity[severity] ?? 'none',
        range: {
            start: { line: startLineNumber, column: startColumn },
            end: { line: endLineNumber, column: endColumn },
        },
    };

    return editorMessage;
}

export function getPrettyMessageText({ message }: { message: MonacoEditorMessage }): string {
    const { status, message: msg, range, source } = message;
    const { line, column } = range.start;
    return `${status}(${line},${column}): ${msg} (${source})`;
}

export function getEmtyCountByStatus(): MonacoEditorMessageCountByStatus {
    return Object.fromEntries(
        monacoEditorMessageStatusOrder.map(status => [status, 0]),
    ) as MonacoEditorMessageCountByStatus;
}

export function getMonacoEditorMessageSummary({
    messages,
}: {
    messages: MonacoEditorMessage[];
}): MonacoEditorMessageSummary {
    const summary: MonacoEditorMessageSummary = {
        summaryStatus: 'none',
        totalCount: 0,
        countByStatus: getEmtyCountByStatus(),
    };
    for (const message of messages) {
        summary.totalCount += 1;
        summary.countByStatus[message.status] += 1;

        const messageIndex = monacoEditorMessageStatusOrderIndexes[message.status];
        const summaryIndex = monacoEditorMessageStatusOrderIndexes[summary.summaryStatus];
        if (messageIndex < summaryIndex) {
            summary.summaryStatus = message.status;
        }
    }

    return summary;
}

export function getEmtyIndexesByStatus(): MessageIndexesByStatus {
    return Object.fromEntries(
        monacoEditorMessageStatusOrder.map(status => [status, new Set()]),
    ) as MessageIndexesByStatus;
}

export function getMessageIndexesByStatus({ messages }: { messages: MonacoEditorMessage[] }): MessageIndexesByStatus {
    const messageIndexesByStatus: MessageIndexesByStatus = getEmtyIndexesByStatus();

    for (let i = 0; i < messages.length; i++) {
        const message = messages[i];
        messageIndexesByStatus[message.status].add(i);
    }

    return messageIndexesByStatus;
}
