import type { editor, IDisposable } from 'monaco-editor';
import type { RefObject } from 'react';
import type { MonacoEditorValidationInfo } from './message';

export const getDefaultEditorOptions: () => editor.IStandaloneEditorConstructionOptions = () => ({
    scrollBeyondLastLine: false,
    scrollbar: {
        horizontal: 'hidden',
        horizontalScrollbarSize: 0,
        vertical: 'hidden',
        verticalScrollbarSize: 0,
        alwaysConsumeMouseWheel: false,
    },
    lineNumbers: 'off',
    minimap: { enabled: false },
    folding: false,
    foldingHighlight: false,
    occurrencesHighlight: false,
    renderLineHighlight: 'none',
});

export const MIN_ROWS = 1;
export const MAX_ROWS = 30;

export interface ChangeValidateParams {
    validationInfo: MonacoEditorValidationInfo;
}

/**
 * свойства, связанные непосредственно с текстом
 */
export type MonacoEditorTextProps = {
    /**
     * текст для редактора
     */
    value: string;

    /**
     * обновление текста, при отсутствии автоматом устанавливается режим чтения
     */
    onUpdate?(v: string): void;

    /**
     * явно устанавливает редактор в режим чтения, даже если есть onUpdate
     */
    readonly?: boolean;
};

export type MonacoEditorLayoutProps = {
    /**
     * строгое количество строк
     */
    rows?: number;

    /**
     * максимально возможное количество строк
     */
    minRows?: number;

    /**
     * минимально возможное количество строк
     */
    maxRows?: number;

    /**
     * автоматическое количество строк, подстраивается под высоту внешнего контейнера, по умолчанию true
     */
    isAutoRows?: boolean;

    /**
     * автоматическое растягивание области редактора на всю высоту доступного контейнера, по умолчанию false
     */
    isAutoMaxHeight?: boolean;
};

export type MonacoEditorThemeProps = {
    /**
     * название темы, если не указана, то подстраивается под тему common
     */
    theme?: string;
};

export type MonacoEditorViewProps = {
    /**
     * пользовательский класс, навешивается на корень редактора
     */
    className?: string;

    /**
     * отрисовка рамки вокруг редактора, по умолчанию true
     */
    showBorder?: boolean;
};

export type MonacoEditorProps = MonacoEditorLayoutProps &
    MonacoEditorTextProps &
    MonacoEditorThemeProps &
    MonacoEditorViewProps & {
        /**
         * uri (https://datatracker.ietf.org/doc/html/rfc3986) используется для определения модели текста
         * в веб можно использовать uri по внутренним соглашениям, затем использовать для lsp
         */
        uri?: string;

        /**
         * опции monaco, при указании затирают уже определенные поля, например value
         */
        options?: editor.IStandaloneEditorConstructionOptions;

        /**
         * текст, который будет показан до загрузки редактора
         */
        placeholder?: string;

        /**
         * явное указание, что редактор в процессе загрузки (например ждёт загрузки воркеров)
         */
        isLoading?: boolean;

        /**
         * хук, срабатывает при ошибках валидации
         */
        onChangeValidate?(params: ChangeValidateParams): void;

        /**
         * ссылка для получения инстанса редактора и апи
         */
        forwardedRef?: RefObject<EditorRef>;

        /**
         * первоначальная настройка редактора
         */
        onSetup?(editor: editor.IStandaloneCodeEditor): IDisposable;
    };

export interface EditorRef {
    getRawEditor(): editor.IStandaloneCodeEditor | undefined;
    getMonacoEditorApi(): typeof editor;
}
