import {isEmpty} from './utils';

import ajaxErrorModule from './modules/ajaxError';
import ajaxInprogressModule from './modules/ajaxInprogress';
import exceptionsModule from './modules/exceptions';
import memoryUsageModule from './modules/memoryUsage';
import networkModule from './modules/network';
import tabsDetectionModule from './modules/tabsDetection';
import visibilityModule from './modules/visibility';

import ajaxStat from './stat_helpers/ajax';
import executionStat from './stat_helpers/execution';
import intervalStat from './stat_helpers/interval';
import {mark, measure, measureNative} from './stat_helpers/measure';
import promiseStat from './stat_helpers/promise';

import * as validation from './lib/validation';

import {Counter} from './Counter';
import {Histogram} from './Histogram';

const SEND_INTERVAL = 5e3;
// const SEND_TIMEOUT = 2e3;
const tabId = Date.now() + '_' + Math.round(Math.random() * 1e6);

class StatsManager {
    static getClassName() {
        return 'StatsManager';
    }

    private counters = new Map<any, any>();
    private histograms = new Map<any, any>();

    private options: {[key: string]: string} = {};
    private queue: any[] = [];
    private tags = new Map<any, any>();

    // experimental
    private firstOnly = true;

    constructor() {
        this.send = this.ajax(this.send.bind(this), 'requests.stats.push').bind(this);
        this.send = this.interval(this.send.bind(this), 'processing.stats.throttling').bind(this);
    }

    config(options: {[key: string]: any} = {}) {
        const {
            // core
            debug = false,

            // server
            itype = 'yasmagentbrowser',
            path = '/push/',

            // modules
            ajax = false,
            exceptions = false,
            memoryUsage = false,
            network = false,
            tabs = false,
            visibility = true
        } = options;

        this.options = {
            ...this.options,

            // core
            debug,

            // server
            itype,
            path,

            // modules
            ajax,
            exceptions,
            memoryUsage,
            network,
            tabs,
            visibility
        };

        return this;
    }

    setUser(value: string) {
        this.setTag('user', value);

        return this;
    }

    setMetricKey(value: string) {
        this.setTag('metric_key', value);

        return this;
    }

    setVisibility(value: string) {
        this.setTag('visibility', value);

        return this;
    }

    rejectFirstOnly() {
        this.firstOnly = false;
    }

    isFirstOnlySupport() {
        return this.firstOnly;
    }

    setTag(name: string, value: string) {
        if (this.tags.get(name) !== value) {
            this.take();
        }

        if (validation.tag(value)) {
            this.tags.set(name, value);
        } else {
            console.warn(`Metrics tag ${name}=${value} is not valid`);
        }

        return this;
    }

    removeTag(name: string) {
        this.tags.delete(name);
        this.take();

        return this;
    }

    getTags() {
        const tags: {[key: string]: any} = {};

        this.tags.forEach((v, k) => {
            tags[k] = v;
        });

        return tags;
    }

    take() {
        const values = this.getYasmSignals();
        this.reset();

        if (isEmpty(values)) {
            return this.queue;
        }

        const tags = this.getTags();

        if (!tags.itype) {
            return this.queue;
        }

        const msg = {
            tags,
            values
        };

        this.queue.push(msg);

        return this.queue;
    }

    install() {
        if (!this.options.path) {
            throw new Error('Path for yasm push is not defined');
        }

        if (this.options.itype) {
            this.setTag('itype', this.options.itype);
        }

        if (this.options.visibility) {
            visibilityModule.install(this);
        }

        if (this.options.ajax) {
            ajaxErrorModule.install(this);
            ajaxInprogressModule.install(this);
        }

        if (this.options.exceptions) {
            exceptionsModule.install(this);
        }

        if (this.options.memoryUsage) {
            memoryUsageModule.install(this);
        }

        if (this.options.network) {
            networkModule.install(this);
        }

        if (this.options.tabs) {
            tabsDetectionModule.install(this);
        }

        setTimeout(this.send, SEND_INTERVAL);
    }

    send() {
        const {debug, path} = this.options;
        const data = this.take();
        this.queue = [];

        const promise = Promise.resolve();

        if (!isEmpty(data) && !debug) {
            promise
                .then(() => {
                    const controller = new AbortController();

                    const request = fetch(path, {
                        body: JSON.stringify(data),
                        headers: {
                            'X-Golovan-Push-Request': tabId
                        },
                        method: 'POST',
                        signal: controller.signal // timeout: SEND_TIMEOUT,
                    });

                    // timeout implementation
                    setTimeout(() => {
                        controller.abort();
                    }, SEND_INTERVAL);

                    return request;
                })
                .catch(() => {
                    /* not catch yet*/
                });
        }

        promise.then(() => setTimeout(this.send, SEND_INTERVAL), () => setTimeout(this.send, SEND_INTERVAL));

        return promise;
    }

    reset() {
        this.counters.clear();
        this.histograms.clear();
    }

    @validation.signal
    incrementCounter(key: string, aggregationType: string, value: number = 1) {
        this.getCounter(key, aggregationType).add(value);
        return this;
    }

    @validation.signal
    setCounterValue(key: string, aggregationType: string, value: number) {
        this.getCounter(key, aggregationType).set(value);
        return this;
    }

    @validation.signal
    addSample(key: string, value: number) {
        this.getHistogram(key).add(value);
        return this;
    }

    getYasmSignals() {
        const metrics = [];

        for (const [key, meter] of this.iterateMetrics()) {
            const [suffix, val] = meter.getYasmFormat();

            metrics.push({
                name: `${key}_${suffix}`,
                val
            });
        }

        return metrics;
    }

    *iterateMetrics() {
        yield* this.histograms.entries();
        yield* this.counters.entries();
    }

    getCounter(key: string, aggregationType?: string) {
        if (!this.counters.has(key)) {
            this.counters.set(key, new Counter(aggregationType));
        }

        return this.counters.get(key);
    }

    getHistogram(key: string, HistogramClass = Histogram) {
        if (!this.histograms.has(key)) {
            this.histograms.set(key, new HistogramClass());
        }

        return this.histograms.get(key);
    }

    ajax(...args: any[]) {
        return ajaxStat(this, ...args);
    }

    promise(...args: any[]) {
        return promiseStat(this, ...args);
    }

    execution(...args: any[]) {
        return executionStat(this, ...args);
    }

    interval(...args: any[]) {
        return intervalStat(this, ...args);
    }

    mark(signal: any) {
        return mark(signal);
    }

    measure(signal: any) {
        return measure(this, signal);
    }

    measureNative(signal: any, from: any, options?: any) {
        return measureNative(this, signal, from, options);
    }
}

export {StatsManager};
