import {interval} from 'rxjs';
import {filter, map, pairwise} from 'rxjs/operators';

import MemoryUsage from './lib/MemoryUsage';

const TAKE_SNAPSHOT_INTERVAL = 5e3;

declare var performance: any;

function isPerformanceMemorySupport() {
    return performance && !!performance.memory;
}

function takeSnapshot() {
    const {totalJSHeapSize, usedJSHeapSize, jsHeapSizeLimit} = performance.memory;
    return new MemoryUsage(totalJSHeapSize, usedJSHeapSize, jsHeapSizeLimit);
}

function toMB(value: number) {
    return value / (1024 * 1024);
}

const memoryUsage = interval(TAKE_SNAPSHOT_INTERVAL).pipe(
    map(takeSnapshot),
    pairwise(),
    filter(([prev, last]) => !last.isSame(prev)),
    map(snapshots => snapshots[1]),
);

export default {
    install(statsManager: any) {
        if (!isPerformanceMemorySupport()) {
            return; // INFO: For memory usage monitoring use Chrome with --enable-precise-memory-info flag
        }

        memoryUsage.subscribe(snapshot => {
            statsManager.addSample('performance.memory.used', toMB(snapshot.used));
            statsManager.addSample('performance.memory.total', toMB(snapshot.total));
            statsManager.addSample('performance.memory.limit', toMB(snapshot.limit));
        });
    },
};
