from infra.unbound import helpers

import yatest.common

import json
import logging
import os
import subprocess

UNBOUND_CONTROL_BINARY_ARCADIA_PATH = "contrib/tools/unbound/unbound-control/unbound-control"


class UnboundControl(object):
    def __init__(
        self, *,
        workdir,
        unbound_config,
    ):
        self.workdir = workdir

        self._init_config(unbound_config)
        self._init_binary()

    def _init_config(self, unbound_config):
        self._unbound_ctl_directory = self._make_ctl_directory()
        self._unbound_config_path = os.path.join(
            self._unbound_ctl_directory,
            "unbound.conf"
        )

        if isinstance(unbound_config, dict):
            logging.debug(f"Dump unbound config to {self._unbound_config_path}")
            logging.debug(f"Config content:\n{json.dumps(unbound_config, indent=2)}")
            with open(self._unbound_config_path) as fout:
                fout.write(helpers.print_unbound_config(unbound_config))
        elif isinstance(unbound_config, os.PathLike) or isinstance(unbound_config, str):
            logging.debug(f"Symlink unbound config: source {unbound_config}, destination {self._unbound_config_path}")
            os.symlink(unbound_config, self._unbound_config_path)
        else:
            raise Exception(f"unbound_config has unknown type: {type(unbound_config)}")

    def _init_binary(self):
        unbound_ctl_binary_arcadia_path = UNBOUND_CONTROL_BINARY_ARCADIA_PATH
        self._unbound_ctl_binary_path = yatest.common.binary_path(unbound_ctl_binary_arcadia_path)

    def _make_ctl_directory(self):
        def get_path(index):
            return os.path.abspath(
                os.path.join(
                    self.workdir,
                    f"unbound_control_{index}"
                )
            )

        index = 0
        while os.path.exists(get_path(index)):
            index += 1
        directory = get_path(index)
        os.makedirs(directory, exist_ok=False)
        return directory

    def _run_cmd(self, cmd_type):
        proc = subprocess.run(
            [self._unbound_ctl_binary_path, "-c", self._unbound_config_path, cmd_type],
            cwd=self._unbound_ctl_directory,
            universal_newlines=True,
            capture_output=True,
        )

        with open(os.path.join(self._unbound_ctl_directory, "unbound-control.stderr"), "a") as stderr:
            stderr.write(proc.stderr)

        with open(os.path.join(self._unbound_ctl_directory, "unbound-control.stdout"), "a") as stdout:
            stdout.write(proc.stdout)

        proc.check_returncode()

        return proc.stdout

    def stop(self):
        return self._run_cmd("stop")

    def reload(self):
        return self._run_cmd("reload")

    def prj_outbound_stats_noreset(self):
        return self._run_cmd("prj_outbound_stats_noreset")
