from infra.unbound import helpers
from infra.unbound.unbound_control import UnboundControl

import yatest.common
from yatest.common import network
import yt.wrapper as yt

import library.python.cores as cores

import copy
import json
import logging
import os
import requests
import signal
import subprocess
import time


UNBOUND_BINARY_ARCADIA_PATH = "contrib/tools/unbound/unbound/unbound"

UNBOUND_CONFIG_TEMPLATE = {
    "server": {},
    "remote-control": {},
}

YP_DNS_CONFIG_TEMPLATE = {
    "LoggerConfig": {"Path": ""},
    "BackupLoggerConfig": {"Path": ""},
    "ConfigUpdatesLoggerConfig": {"Path": ""},
    "HttpServiceConfig": {},
    "DynamicZonesConfig": {
        "ReplicaLoggerConfig": {"Path": ""},
    },
    "YPClusterConfigs": [],
    "YPReplicaConfig": {},
    "Zones": [],
}


class UnboundInstance(object):
    def __init__(
        self,
        workdir=None,
        unbound_config=None,
        yp_dns_config=None,
    ):
        self._port_manager = network.PortManager()

        self._unbound_binary_path = None
        self._unbound_config_patch = unbound_config
        self._yp_dns_config_patch = yp_dns_config
        self._unbound_process = None

        self.workdir = workdir
        self._unbound_config = None
        self._yp_dns_config = None

    def generate_configs(self):
        unbound_config = copy.deepcopy(UNBOUND_CONFIG_TEMPLATE)
        yp_dns_config = copy.deepcopy(YP_DNS_CONFIG_TEMPLATE)

        self.unbound_host = "::1"
        self.unbound_port = self._port_manager.get_port()
        self.unbound_address = f"[{self.unbound_host}]:{self.unbound_port}"
        unbound_config["server"]["port"] = self.unbound_port

        if self._unbound_config_patch is not None and "remote-control" in self._unbound_config_patch:
            self.unbound_control_port = self._port_manager.get_port()
            self.unbound_control_address = f"[{self.unbound_host}]:{self.unbound_control_port}"
            unbound_config["remote-control"]["control-port"] = self.unbound_control_port

        self.yp_dns_host = "::1"
        self.yp_dns_port = self._port_manager.get_port()
        self.yp_dns_address = f"[{self.yp_dns_host}]:{self.yp_dns_port}"
        yp_dns_config["HttpServiceConfig"]["Port"] = self.yp_dns_port

        if self._unbound_config_patch is not None:
            unbound_config = yt.common.update(unbound_config, copy.deepcopy(self._unbound_config_patch))

        self._patch_ports(unbound_config)

        if self._yp_dns_config_patch is not None:
            yp_dns_config = yt.common.update(yp_dns_config, copy.deepcopy(self._yp_dns_config_patch))

        return unbound_config, yp_dns_config

    def _patch_ports(self, unbound_config):
        for stub_zone in helpers.iter_attribute(unbound_config.get("stub-zone")):
            helpers.transform_attribute(stub_zone, "stub-addr", lambda addr: addr.format(unbound_port=self.unbound_port))

    def _make_unbound_directory(self):
        def get_path(index):
            return os.path.abspath(
                os.path.join(
                    self.workdir,
                    f"unbound_{index}"
                )
            )

        index = 0
        while os.path.exists(get_path(index)):
            index += 1
        directory = get_path(index)
        os.makedirs(directory, exist_ok=False)
        return directory

    def _check_process_alive(self, proc, wait_exit=None):
        if proc is None:
            return False
        if wait_exit is not None:
            try:
                if isinstance(wait_exit, (int, float)) and not isinstance(wait_exit, bool):
                    timeout = wait_exit
                else:
                    timeout = None
                proc.wait(timeout=timeout)
            except subprocess.TimeoutExpired:
                logging.debug("Timeout expired while waiting for process")
                pass
        return proc.poll() is None

    def _assert_unbound_alive(self):
        assert self._check_process_alive(self._unbound_process), \
            f"Unbound process unexpectedly terminated with error code {self._unbound_process.returncode}"

    def _log_addresses(self):
        address_variables = {
            "Unbound": "unbound_address",
            "Unbound Control": "unbound_control_address",
            "YP DNS": "yp_dns_address",
        }
        for title, address_variable_name in address_variables.items():
            if hasattr(self, address_variable_name):
                logging.info("{} address: {}".format(
                    title, getattr(self, address_variable_name)))

    def _start(self, unbound_binary_path):
        assert not self._check_process_alive(self._unbound_process), \
            f"Unbound has already been started with pid {self._unbound_process.pid}"

        self._unbound_directory = self._make_unbound_directory()

        if unbound_binary_path is None:
            unbound_binary_arcadia_path = UNBOUND_BINARY_ARCADIA_PATH
            unbound_binary_path = yatest.common.binary_path(unbound_binary_arcadia_path)
        self._unbound_binary_path = unbound_binary_path

        self._unbound_config, self._yp_dns_config = self.generate_configs()

        yp_dns_logs_path = os.path.join(self._unbound_directory, "yp_dns_logs")
        config_log_paths = [
            ("LoggerConfig", "Path"),
            ("BackupLoggerConfig", "Path"),
            ("ConfigUpdatesLoggerConfig", "Path"),
            ("DynamicZonesConfig", "ReplicaLoggerConfig", "Path"),
        ]
        for config_log_path in config_log_paths:
            log_path = os.path.join(yp_dns_logs_path, helpers.get_by_path(self._yp_dns_config, config_log_path))
            helpers.set_by_path(self._yp_dns_config, config_log_path, log_path)
            os.makedirs(os.path.dirname(log_path), exist_ok=True)

        self._yp_dns_config_path = os.path.join(self._unbound_directory, "yp_dns_config.json")
        self._dump_yp_dns_config()

        self._unbound_config["server"].setdefault("yp-dns-config-path", self._yp_dns_config_path)

        self._unbound_config_path = os.path.join(self._unbound_directory, "unbound.conf")
        self._dump_unbound_config()

        stderr = open(os.path.join(self._unbound_directory, "unbound.stderr"), "w")
        stdout = open(os.path.join(self._unbound_directory, "unbound.stdout"), "w")

        cmd = [self._unbound_binary_path, "-c", self._unbound_config_path, "-dd"]

        logging.info("Start Unbound server: {}".format(cmd))
        self._unbound_process = subprocess.Popen(
            cmd,
            cwd=self._unbound_directory,
            stdout=stdout,
            stderr=stderr,
        )

        time.sleep(0.5)
        self._assert_unbound_alive()

        logging.info(f"Unbound server has started successfully with pid {self._unbound_process.pid}")
        self._log_addresses()

    def _dump_unbound_config(self):
        with open(self._unbound_config_path, "w") as fout:
            fout.write(helpers.print_unbound_config(self._unbound_config))

    def _dump_yp_dns_config(self):
        with open(self._yp_dns_config_path, "w") as fout:
            json.dump(self._yp_dns_config, fout, indent=4)

    def start(self, unbound_binary_path=None):
        if hasattr(self, "_unbound_control"):
            self._recreate_unbound_control_instance()

        def _start_and_process_error():
            try:
                self._start(unbound_binary_path=unbound_binary_path)
            except Exception as e:
                logging.exception("Failed to start Unbound instance")
                try:
                    self.stop()
                except:
                    logging.exception("Failed to stop Unbound instance after start failure")
                raise e

        _start_and_process_error()

    def _print_coredump(self):
        logging.info(f"Trying to get stacktraces for Unbound (pid={self._unbound_process.pid})")

        gdb_path = yatest.common.runtime.gdb_path()
        logging.debug({
            "binary": self._unbound_binary_path,
            "work dir": self._unbound_directory,
            "GDB": gdb_path,
        })

        core_path = cores.recover_core_dump_file(
            self._unbound_binary_path,
            self._unbound_directory,
            self._unbound_process.pid,
        )

        if core_path:
            logging.info(f"Coredump found at {core_path}")
            backtrace = cores.get_gdb_full_backtrace(self._unbound_binary_path, core_path, gdb_path)
            logging.info(f"Stacktrace:\n{backtrace}")
        else:
            logging.info("No coredump found")

    def _stop_by_signal(self, signal, ensure_alive=True, print_coredump=False):
        if ensure_alive:
            self._assert_unbound_alive()
        elif not self._check_process_alive(self._unbound_process):
            logging.debug("Unbound (pid={self._unbound_process.pid}) is not alive")
            return

        try:
            self._unbound_process.send_signal(signal)
            assert not self._check_process_alive(self._unbound_process, wait_exit=True)
            logging.info(f"Unbound process (pid={self._unbound_process.pid}) has been successfully stopped with signal {signal}")
        except OSError:
            logging.exception(f"Failed to stop Unbound instance (pid={self._unbound_process.pid}) by sending {signal}")
            raise

        if print_coredump:
            self._print_coredump()

    def abort(self, ensure_alive=True, print_coredump=False):
        self._stop_by_signal(signal.SIGABRT, ensure_alive, print_coredump)

    def kill(self, ensure_alive=True, print_coredump=False):
        self._stop_by_signal(signal.SIGKILL, ensure_alive, print_coredump)

    def gracefully_shutdown(self):
        self._assert_unbound_alive()

        if not self.remote_control_enabled:
            raise Exception("Remote control is disabled")

        try:
            resp = self.unbound_control.stop()
            logging.info(f"unbound-control stop cmd output: {repr(resp)}")
        except:
            logging.exception("Failed to stop Unbound instance via unbound-control stop")
            raise

        assert not self._check_process_alive(self._unbound_process, wait_exit=10), \
            f"Unbound process (pid={self._unbound_process.pid}) did not exit after stop command"
        assert self._unbound_process.returncode == 0, \
            f"Unbound unexpectedly exited with code {self._unbound_process.returncode} after stop command"

    def stop(self):
        self._assert_unbound_alive()

        if self.remote_control_enabled:
            try:
                self.gracefully_shutdown()
                logging.info(f"Unbound process (pid={self._unbound_process.pid}) has been gracefully shot down")
            except:
                logging.exception("Failed to gracefully shutdown Unbound. Try to kill with coredump")
                self.abort(ensure_alive=False, print_coredump=True)
                raise
        else:
            self.kill()

        self._unbound_process = None
        self._unbound_config = None
        self._yp_dns_config = None
        self._unbound_directory = None
        self._unbound_binary_path = None

    def reload(self, unbound_config=None, yp_dns_config=None):
        if unbound_config is not None:
            self._unbound_config = unbound_config
            self._dump_unbound_config()

        if yp_dns_config is not None:
            self._yp_dns_config = yp_dns_config
            self._dump_yp_dns_config()

        assert self.unbound_control.reload() == 'ok\n'

    def wait_for_yp_dns_start(self):
        while True:
            self._assert_unbound_alive()
            try:
                pong = self.yp_dns_ping()
                pong.raise_for_status()
                break
            except requests.exceptions.RequestException:
                time.sleep(0.5)
                continue

    def is_yp_dns_ready(self):
        self._assert_unbound_alive()

        try:
            sensors = self.yp_dns_sensors()
        except requests.exceptions.RequestException:
            time.sleep(0.5)
            return False

        ready_sensor = next(filter(
            lambda sensor: sensor["labels"]["sensor"] == "unbound.yp_dns_service.ready",
            sensors["sensors"]
        ))
        return ready_sensor is not None and ready_sensor["value"] == 1

    def wait_for_yp_dns_readiness(self):
        while not self.is_yp_dns_ready():
            pass

    def create_unbound_control(self):
        return UnboundControl(
            workdir=self.workdir,
            unbound_config=self._unbound_config_path,
        )

    def _recreate_unbound_control_instance(self):
        self._unbound_control = self.create_unbound_control()

    @property
    def unbound_config(self):
        return copy.deepcopy(self._unbound_config)

    @property
    def yp_dns_config(self):
        return copy.deepcopy(self._yp_dns_config)

    @property
    def remote_control_enabled(self):
        return self._unbound_config["remote-control"].get("control-enable", False)

    @property
    def unbound_control(self):
        if not hasattr(self, "_unbound_control"):
            self._recreate_unbound_control_instance()
        return self._unbound_control

    def yp_dns_ping(self):
        return requests.get(f"http://{self.yp_dns_address}/ping")

    def yp_dns_sensors(self):
        return requests.get(f"http://{self.yp_dns_address}/sensors/json").json()

    def list_zone_data(self, zone, mode='json'):
        req = {
            "zone": zone,
        }
        r = requests.post(f"http://{self.yp_dns_address}/list_zone_data/{mode}", json=req)
        if mode == 'json':
            return r.json()
        else:
            return r.content
