#!/usr/bin/env python
from __future__ import print_function

import os
import sys
import errno
import argparse
import logging
import subprocess

from releaselib import sandbox
from releaselib import setuppy


SVN_BASE_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
SVN_PROJECT_TAGS_URL_TEMPLATE = 'svn+ssh://arcadia.yandex.ru/arc/tags/infra/vmagent/{}'


def setup_logging():
    logger = logging.getLogger()
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)
    logger.addHandler(console)
    logger.setLevel(logging.INFO)
    return logger


logger = setup_logging()


class SvnError(Exception):
    def __init__(self, message, exception=None):
        self.message = message
        self.exception = exception

    def __str__(self):
        rv = self.message
        e = self.exception
        if e:
            if isinstance(e, subprocess.CalledProcessError):
                rv += '\ncmd: {}\nreturncode: {}\noutput: {}'.format(e.cmd, e.returncode, e.output)
            else:
                rv += '\n{}'.format(e)
        return rv


def run(cmd, input_=None):
    if input_ is None:
        return subprocess.check_output(cmd, stderr=subprocess.STDOUT).strip()
    else:
        p = subprocess.Popen(cmd, stdin=subprocess.PIPE, stderr=subprocess.STDOUT, stdout=subprocess.PIPE)
        return p.communicate(input_)[0].strip()


def create_svn_tag(base_url, tag_url):
    msg = 'Create tag {} from svn url {}'.format(tag_url, base_url)
    cmd = ['svn', 'copy', base_url, tag_url, '-m', msg]
    logger.debug(msg)
    try:
        run(cmd)
    except (subprocess.CalledProcessError, OSError) as e:
        raise SvnError('Failed to create tag', e)


def create_svn_commit(message):
    cmd = ['svn', 'commit', '-m', message]
    logger.debug('Creating svn commit...')
    try:
        r = run(cmd)
    except (subprocess.CalledProcessError, OSError) as e:
        raise SvnError('Failed to create commit', e)
    return r.split()[-1].rstrip('.')


def get_sandbox_oauth_token():
    try:
        with open(os.path.expanduser('~/.sandbox_oauth')) as fd:
            return fd.read().strip()
    except IOError as e:
        if errno.ENOENT == e.errno:
            return None


if __name__ == '__main__':
    token = get_sandbox_oauth_token()
    parser = argparse.ArgumentParser()
    parser.add_argument('--sandbox_oauth_token', default=token,
                        help='Sandbox OAuth token (will be taken from ~/.sandbox_oauth by default)')
    args = parser.parse_args()
    if args.sandbox_oauth_token is None:
        logger.error('Sandbox OAuth token not given and not found in ~/.sandbox_oauth. Please acquire it from https://sandbox.yandex-team.ru/oauth/.')
        sys.exit(1)

    new_version = setuppy.set_or_bump_version(bump='minor')
    version_str = 'stable/{}'.format(new_version)
    logger.info('Commit new version {}'.format(version_str))
    rev = create_svn_commit('Bump infra/vmagent version to {}'.format(version_str))
    logger.info('Committed revision {}'.format(rev))

    sandbox_client = sandbox.SandboxClient(oauth_token=args.sandbox_oauth_token)
    ctx = {
        'checkout_arcadia_from_url': 'arcadia:/arc/trunk/arcadia@{}'.format(rev),
        'arcadia_patch': 'https://paste.yandex-team.ru/592379/text',
        'build_system': 'ya',
        'build_type': 'release',
        'test': True,
        'version': str(new_version),
    }
    logger.info('Creating Sandbox task')
    task_id = sandbox_client.create_task(
        type_name='BUILD_VMAGENT',
        owner='NANNY',
        description='vmagent: {}'.format(new_version),
        ctx=ctx,
        priority=('SERVICE', 'HIGH'),
    )
    logger.info('https://sandbox.yandex-team.ru/task/{}/view'.format(task_id))
