#!/usr/bin/python
import hashlib
import json
import os
import socket
import subprocess
import sys
import time
import traceback
import urlparse

from library.python import resource

import config
import porto
from sandbox.common.proxy import OAuth
from sandbox.common.rest import Client as SandboxClient

SANDBOX_TIMEOUT = 60 * 5
DEFAULT_TASK_OWNER = 'QEMU_BACKUP'
DOWNLOAD_SPEED = 40


def get_image_size(path):
    return os.stat(path).st_size


def get_storage_size(path):
    result = os.statvfs(path)
    block_size = result.f_frsize
    total_blocks = result.f_blocks
    total_size = total_blocks * block_size
    return total_size


def get_virtual_size(path):
    process = subprocess.Popen(["qemu-img", "info", path, "--output", "json"], stdout=subprocess.PIPE)
    return json.loads(process.communicate()[0])["virtual-size"]


def log_msg(msg):
    sys.stderr.write(msg + "\n")


def log_trace(*args, **kwargs):
    traceback.print_tb(*args, **kwargs)


class QemuMonError(Exception):
    pass


class SkynetError(Exception):
    pass


def qemu_mon(cmd, timeout=30):
    deadline = time.time() + timeout
    now = time.time()
    exc = None

    while now < deadline:
        try:
            s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            s.settimeout(deadline - now)
            s.connect(config.MONITOR_PATH)
            s.recv(1024)
            s.sendall(cmd + '\n')
            s.recv(1024)
            s.close()
            return

        except Exception as e:
            exc = e
            time.sleep(0.5)
            pass

        now = time.time()

    raise QemuMonError("Socket timeout: {}".format(exc))


def get_file_md5(path):
    if not os.access(path, os.R_OK):
        return None

    m = hashlib.md5()
    with open(path) as f:
        s = f.read(1048576)
        while len(s) > 0:
            m.update(s)
            s = f.read(1048576)

    return m.hexdigest()


def gen_vnc_password():
    with open("/dev/urandom", "r") as f:
        m = hashlib.sha256()
        m.update(f.read(64))
        return m.hexdigest()[0:8]


def purge_dir_contents(dirpath):
    try:
        subprocess.check_output(["sudo", "find", dirpath, "-mindepth", "1", "-delete"])

    except subprocess.CalledProcessError as e:
        log_msg("error purging {}: {} {}".format(dirpath, e.returncode, e.output))


def drop_delta():
    return remove_disk(config.DELTA_PATH)


def remove_disk(path):
    if not path:
        return

    try:
        os.remove(path)
    except OSError as e:
        if e.errno != 2:
            raise e


def get_qemu_container(new_one=True):
    c = porto.Connection(timeout=30)
    name = 'self/qemu'
    try:
        r = c.Find(name)

        if not new_one:
            return r

        r.Destroy()

    except porto.exceptions.ContainerDoesNotExist:
        if not new_one:
            return None

    r = c.Create(name)

    return r


class SkynetContainer(object):

    def __init__(self, timeout=300):
        self._timeout = timeout

    def __enter__(self):
        c = porto.Connection(timeout=30)

        try:
            r = c.Find("skynet")
            r.Destroy()
        except porto.exceptions.ContainerDoesNotExist:
            pass

        try:
            os.remove(config.STORAGE_PATH + "/skynet.started")
        except OSError as e:
            if e.errno != 2:
                raise e

        r = c.Create("skynet")
        r.SetProperty("net", "L3 eth0")
        r.SetProperty("hostname", config.HOSTNAME)
        r.SetProperty("isolate", False)
        r.SetProperty("umask", "0")
        r.SetProperty("env[SKYNET_HOSTNAME]", config.HOSTNAME)
        r.SetProperty("env[SKYNET_IP]", config.SKYNET_IP)
        r.SetProperty("env[SKYNET_AUX_IP]", config.SKYNET_AUX_IP.split()[0])
        r.SetProperty("command", "sudo -E /bin/bash -x {}".format(config.SKYNET_LAUNCHER_PATH))

        ip_setting = "eth0 {};".format(config.SKYNET_IP)

        for ip in config.SKYNET_AUX_IP.split():
            ip_setting += "eth0 {};".format(ip)

        r.SetProperty("ip", ip_setting)

        r.Start()

        deadline = time.time() + self._timeout

        while time.time() < deadline:
            if r.Wait(timeout=1000):
                res = r.Get(["exit_code", "stderr"])
                r.Destroy()
                raise SkynetError("Skynet exit code {} : {}"
                                  .format(res["exit_code"], res["stderr"]))

            if os.access(config.STORAGE_PATH + "/skynet.started", os.F_OK):
                return

        r.Destroy()

        log_msg("Skynet start timeout")
        raise SkynetError("Skynet start timeout")

    def __exit__(self, exc_type, exc_val, exc_tb):
        c = porto.Connection(timeout=30)
        c.Destroy("skynet")

        return False


def get_resource(rb_torrent, path):

    try:
        subprocess.check_output(["sky", "get", "-u", "-d", path, rb_torrent],
                                stderr=subprocess.STDOUT)

    except subprocess.CalledProcessError as e:
        raise SkynetError("sky get error: {}, {}".format(e.returncode, e.output))


def ping_vm(timeout=1):
    try:
        subprocess.check_output(
            [
                "/bin/ping6", "-c", "1", "-W",
                "{}".format(int(timeout)), config.VM_IP
            ]
        )

        return True

    except subprocess.CalledProcessError:
        return False


def connect_vm(timeout=0.1):
    try:
        sk = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        sk.settimeout(0.1)
        sk.connect((config.VM_IP, 22))
        sk.close()
        return True

    except Exception:
        return False


def share_resource(path):
    try:
        cwd, file_path = os.path.split(path)
        result = subprocess.check_output(['sky', 'share', '-d', cwd, file_path])
        return result.strip()
    except subprocess.CalledProcessError as e:
        raise SkynetError('sky share error: {}, {}'.format(e.returncode, e.output))


def get_sandbox_task_url(client, task_id):
    return urlparse.SplitResult(
        scheme='https',
        netloc=client.host,
        path=str(client.task[task_id]),
        query='',
        fragment=''
    ).geturl()


def run_sandbox_merge_task(image_url, delta_url, size, dc):
    """
    :type image_url: str
    :type delta_url: str
    :type size: int
    :type dc: str
    """
    robot_token = resource.find('/secrets/sandbox_token').rstrip()
    client = SandboxClient(auth=OAuth(robot_token), total_wait=SANDBOX_TIMEOUT)
    kill_timeout = max(calculate_time(size, DOWNLOAD_SPEED) * 2, 3 * 60 * 60)
    task_id = client.task.create({
        'type': 'MERGE_QEMU_IMAGE',
        'owner': DEFAULT_TASK_OWNER,
        'description': 'Merge delta to original QEMU image',
        'priority': ('SERVICE', 'HIGH'),
        'custom_fields': [
            {
                'name': 'image_url',
                'value': image_url,
            },
            {
                'name': 'delta_url',
                'value': delta_url
            }
        ],
        'kill_timeout': kill_timeout,
        'requirements': {
            'disk_space': size * 2,
            'client_tags': 'LINUX & {}'.format(dc)
        }
    })['id']
    client.batch.tasks.start.update([task_id])
    return get_sandbox_task_url(client, task_id)


def run_sandbox_upload_task(image_url, size, dc):
    """
    :type image_url: str
    :type size: int
    :type dc: str
    """
    robot_token = resource.find('/secrets/sandbox_token').rstrip()
    client = SandboxClient(auth=OAuth(robot_token), total_wait=SANDBOX_TIMEOUT)
    kill_timeout = max(calculate_time(size, DOWNLOAD_SPEED), 3 * 60 * 60)
    task_id = client.task.create({
        'type': 'COPY_VM_IMAGE',
        'owner': DEFAULT_TASK_OWNER,
        'description': 'Share Qemu image with Sandbox',
        'priority': ('SERVICE', 'HIGH'),
        'custom_fields': [
            {
                'name': 'image_url',
                'value': image_url,
            },
        ],
        'kill_timeout': kill_timeout,
        'requirements': {
            'disk_space': size + 1024 ** 3,
            'client_tags': 'GENERIC & {}'.format(dc)
        }
    })['id']
    client.batch.tasks.start.update([task_id])
    return get_sandbox_task_url(client, task_id)


def calculate_time(size, download_speed):
    """
    :type size: int
    :type download_speed: int
    """
    mb_speed = download_speed / 8
    size_mb = size / 1024 ** 2
    return size_mb / mb_speed
