#!/bin/bash -x

# Id variables:
#
# HOST : instance host
# PORT : instance port
# LOGIN_PASSWORD : default user credentails
# 
# Resource limits:
# 
# VCPU_NR : vcpu numbert
# MEM : vm memory
# DISK_SIZE : disk size
#
# Paths:
#
# IMG_PATH : path to vm disk image resource
# CURRENT_PATH : path to delta disk
# STORAGE_PATH : path to persistent storage
# 
# Net settings:
#
# TAP_DEV : host tap device name
# TAP_LL : host tap device link-local address
# VM_IP : vm ipv6 address


: ${HOST:="anon_qemuvm"}
: ${PORT:=7255}
: ${LOGIN_PASSWORD:="P@ssw0rd"}
: ${VCPU_NR:=1}
: ${MEM:="1G"}
: ${DISK_SIZE:="10485760"}
: ${MODE:="gencfg"}

MEM_MB=$(echo $MEM | sed 's:\([0-9]\+\)M:\1:g')
MEM=${MEM_MB}M

# Warn user about inconsistent qcow2 to be created
VIRTUAL_DISK_SIZE=$(qemu-img info $IMG_PATH --output json | python -c "import json; import sys; print json.loads(sys.stdin.read())[\"virtual-size\"]")

if [[ $VIRTUAL_DISK_SIZE -gt $DISK_SIZE ]]; then
    echo "WARNING: REQUESTED VIRTUAL DISK IS SMALLER THAN ITS BASE"
fi

VM_HOSTNAME=$HOST
WORKDIR=$PWD

: ${IMG_PATH:?"No image path supplied"}
: ${STORAGE_PATH:?"No storage path supplied"}

: ${INSTANCE_ID=$(jq .configurationId -r ./dump.json)}

: ${CURRENT_PATH:?"No delta path supplied"}

: ${NETCONF_PATH:=$STORAGE_PATH"/netconf"}
: ${LOG_DIR:=$STORAGE_PATH"/logs"}

mkdir $LOG_DIR || true

#FIXME: Shoud be rotated in chroot
: ${SERIAL_LOG:=$LOG_DIR"/serial.log"}

VM_MAC="52:54:00:12:34:56"

: ${TAP_LL:?"No link-local address supplied"}
: ${TAP_DEV:?"No tap device name supplied"}
: ${VM_IP:?"No vm ipv6 address supplied"}
: ${VM_AUX_IP:?"No aux (fb) vm ipv6 address supplied"}

read -r -a ITAGS <<< `echo "$BSCONFIG_ITAGS" | sed -e "s/'//g"`

# Uncomment to let instance drop VM state on start

MON_PORT=$((PORT+1)) # PORT + 1 - for qemu monitor
SERIAL_PORT=$((PORT+2)) # PORT + 2 - for qemu serial
VNC_PORT=$((PORT+3)) # PORT + 3 - for vnc

if [[ -e $NETCONF_PATH ]]; then
    rm -rv $NETCONF_PATH
fi

mkdir $NETCONF_PATH

tee $NETCONF_PATH/meta-data <<EOF
instance-id: iid-$INSTANCE_ID
local-hostname: $VM_HOSTNAME
EOF

if [[ "$MODE" == "gencfg" ]]; then
tee -a $NETCONF_PATH/meta-data <<EOF
network-interfaces: |
    auto eth0
    iface eth0 inet6 static
        address $VM_IP/128
        gateway $TAP_LL
    post-up /bin/echo -e "search search.yandex.net gencfg-c.yandex.net yandex.net\nnameserver 2a02:6b8:0:3400::1\nnameserver 2a02:6b8::1:1\noptions timeout:1 attempts:1" > /etc/resolv.conf
    post-up ip -6 addr replace $VM_AUX_IP/128 dev eth0
    post-up ip -6 addrlabel add prefix 2a02:6b8:f000::/36 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:a00::/56 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2620:10f:d00f::/48 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:1603::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:f07::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:f01::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:c0b::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:c03::/64 dev eth0 label 100 || true
    post-up ip -6 route replace 2a02:6b8::/32 via $TAP_LL dev eth0 mtu 8910 || true
    post-up ip -6 route replace 2620:10f:d000::/44 via $TAP_LL dev eth0 mtu 8910 || true
EOF
elif [[ "$MODE" == "yp" ]]; then
tee -a $NETCONF_PATH/meta-data <<EOF
network-interfaces: |
    auto eth0
    iface eth0 inet dhcp
    iface eth0 inet6 dhcp
    post-up sysctl -w net.ipv6.conf.eth0.accept_ra=2 || true
    post-up sysctl -w net.ipv6.conf.eth0.accept_ra_defrtr=1 || true
    post-up ip -6 addr replace $VM_AUX_IP/128 dev eth0
    post-up ip -6 addrlabel add prefix 2a02:6b8:f000::/36 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:a00::/56 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2620:10f:d00f::/48 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:1603::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:f07::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:f01::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:c0b::/64 dev eth0 label 100 || true
    post-up ip -6 addrlabel add prefix 2a02:6b8:0:c03::/64 dev eth0 label 100 || true
    post-up ip -6 route replace 2a02:6b8::/32 via fe80::1 dev eth0 mtu 8910 || true
    post-up ip -6 route replace 2620:10f:d000::/44 via fe80::1 dev eth0 mtu 8910 || true
EOF
fi


if [[ "$MODE" == "gencfg" ]]; then
tee $NETCONF_PATH/network-config <<EOF
config:
- mac_address: '$VM_MAC'
  type: physical
  name: eth0
  subnets:
  - {address: '$VM_IP/128', gateway: '$TAP_LL', type: 'static'}
- type: nameserver
  address:
  - 2a02:6b8:0:3400::1
  - 2a02:6b8::1:1
  search:
  - search.yandex.net
  - yandex.net
  - gencfg-c.yandex.net
version: 1
EOF
elif [[ "$MODE" == "yp" ]]; then
tee $NETCONF_PATH/network-config <<EOF
config:
- mac_address: '$VM_MAC'
  type: physical
  name: eth0
  subnets:
  - {type: 'dhcp4'}
  - {type: 'dhcp6'}
version: 1
EOF
fi

NET="-netdev tap,id=qemu_net,ifname=$TAP_DEV,script=no -device virtio-net-pci,netdev=qemu_net,mac=$VM_MAC"


tee $NETCONF_PATH/user-data <<EOF
#cloud-config
growpart:
  mode: auto
  devices: ['/']
resize_rootfs: True
runcmd:
  - ['sgdisk', '/dev/vda', '--attributes=1:set:2']
manage_etc_hosts: False
hostname: "$VM_HOSTNAME"
fqdn: "$VM_HOSTNAME"
disable_root: False
apt:
  preserve_sources_list: True
apt_preserve_sources_list: true
power_state:
  mode: reboot
  delay: now
  message: 'Reboot instance'
  timeout: 60
  condition: true
write_files:
  - path: /etc/sysctl.d/10-qemu-rtc-accept-ra-eth0.conf
    permissions: '0644'
    owner: root:root
    content: |
      net.ipv6.conf.eth0.accept_ra = 2
  - path: /etc/hosts
    permissions: '0644'
    owner: root:root
    content: |
      127.0.0.1 localhost
      ::1 localhost ip6-localhost ip6-loopback
      fe00::0 ip6-localnet
      ff00::0 ip6-mcastprefix
      ff02::1 ip6-allnodes
      ff02::2 ip6-allrouters
      ff02::3 ip6-allhosts
  - path: /etc/network/interfaces.d/50-qemu-rtc-accept-ra.cfg
    permissions: '0755'
    owner: root:root
    content: |
      auto eth0
      iface eth0 inet6 manual
      post-up sysctl -w net.ipv6.conf.eth0.accept_ra=2 || true
  - path: /etc/network/interfaces.d/50-fastbone.cfg
    permissions: '0755'
    owner: root:root
    content: |
      auto eth0
      iface eth0 inet6 manual
      post-up ip -6 addr replace $VM_AUX_IP/128 dev eth0
      post-up ip -6 addrlabel add prefix 2a02:6b8:f000::/36 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:a00::/56 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2620:10f:d00f::/48 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:1603::/64 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:f07::/64 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:f01::/64 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:c0b::/64 dev eth0 label 100 || true
      post-up ip -6 addrlabel add prefix 2a02:6b8:0:c03::/64 dev eth0 label 100 || true
EOF

if [[ "$MODE" == "yp" ]]; then
tee -a $NETCONF_PATH/user-data <<EOF
  - path: /etc/network/interfaces.d/50-backbone-jumbo.cfg
    permissions: '0755'
    owner: root:root
    content: |
      auto eth0
      iface eth0 inet6 manual
      post-up ip -6 route replace 2a02:6b8::/32 via fe80::1 dev eth0 mtu 8910 || true
      post-up ip -6 route replace 2620:10f:d000::/44 via fe80::1 dev eth0 mtu 8910 || true
EOF
fi

if [[ "$MODE" == "gencfg" ]]; then
tee -a $NETCONF_PATH/user-data <<EOF
  - path: /etc/network/interfaces.d/50-backbone-jumbo.cfg
    permissions: '0755'
    owner: root:root
    content: |
      auto eth0
      iface eth0 inet6 manual
      post-up ip -6 route replace 2a02:6b8::/32 via $TAP_LL dev eth0 mtu 8910 || true
      post-up ip -6 route replace 2620:10f:d000::/44 via $TAP_LL dev eth0 mtu 8910 || true
  - path: /etc/rtc-gencfg.group
    permissions: '0444'
    owner: root:root
    content: ${ITAGS[0]}
EOF
fi


if [[ -e ./authorized_keys ]]; then
    echo "ssh_authorized_keys:" >> $NETCONF_PATH/user-data
    sed 's/^/  - /g' ./authorized_keys >> $NETCONF_PATH/user-data
fi

if [ -r "/dev/kvm" ] && [ -w "/dev/kvm" ]; then
    KVM_FLAG="--enable-kvm -cpu host"
else
    KVM_FLAG=""
fi

CHARDEV+="-chardev socket,id=monsk,path=./mon.sock,server,nowait -mon monsk -monitor none "

CHARDEV+="-chardev file,id=serial_log,path=$SERIAL_LOG -serial chardev:serial_log"

DISPLAY+="-usbdevice tablet -vnc unix:./vnc.sock,password -vga std "

DRIVE_PATH=""
if [[ $IMAGE_TYPE == "RAW" ]]; then
    DRIVE_PATH=$IMG_PATH
    qemu-img resize $IMG_PATH $SIZE_FOR_RAW
else
    DRIVE_PATH=$CURRENT_PATH
    if [[ ! -e $CURRENT_PATH ]]; then
        qemu-img create -f qcow2 -b $IMG_PATH $CURRENT_PATH $DISK_SIZE
    fi
fi

# Redirect tcp connections to unix sockets
socat TCP6-LISTEN:$VNC_PORT,reuseaddr,fork UNIX-CONNECT:./vnc.sock &
socat UDP6-LISTEN:$VNC_PORT,reuseaddr,fork UNIX-CONNECT:./vnc.sock &

/usr/local/bin/qemu-system-x86_64 \
$KVM_FLAG \
-m "$MEM" -smp "$VCPU_NR" \
-nographic \
-nodefaults \
-device virtio-rng-pci \
-drive file="$DRIVE_PATH",if=virtio,cache=none \
-drive file=fat:"$NETCONF_PATH",if=virtio,file.label=cidata,readonly=on \
$CHARDEV \
$DISPLAY \
$NET

RET=$?

pkill -9 -P $$

exit $RET
