import collections
import getpass
import operator

from infra.vmagent.src.vmagent_pb import vmset_pb2
from infra.vmagent.src.vmctl import api, helpers
from infra.vmagent.src.vmctl.actions.list_helpers import ListField, int_to_str_gigabytes
from tabulate import tabulate


DEV_SEGMENT = 'dev'


def get_limit(value):
    def wrapped(obj, **kwargs):
        segment = kwargs['segment']
        return operator.attrgetter(value)(obj.limits.per_segment[segment])
    return wrapped


def get_usage(value):
    def wrapped(obj, **kwargs):
        segment = kwargs['segment']
        return operator.attrgetter(value)(obj.usage.per_segment[segment])
    return wrapped


def get_disk_limit(storage):
    def wrapped(obj, **kwargs):
        segment = kwargs['segment']
        return obj.limits.per_segment[segment].disk_per_storage.get(storage) or 0
    return wrapped


def get_disk_usage(storage):
    def wrapped(obj, **kwargs):
        segment = kwargs['segment']
        return obj.usage.per_segment[segment].disk_per_storage.get(storage) or 0
    return wrapped


def get_qemu_disk_size(storage):
    def wrapped(obj, **kwargs):
        for vol in obj.spec.qemu.volumes:
            if vol.name == '/qemu-persistent' and vol.storage_class == storage:
                return vol.capacity
        return 0
    return wrapped


AVAILABLE_FIELDS = collections.OrderedDict([
    ('name', ListField('Name', value_getter=lambda obj, **kwargs: kwargs['name'])),
    ('location', ListField('Location', value_getter=lambda obj, **kwargs: kwargs['cluster'])),
    ('segment', ListField('Segment', value_getter=lambda obj, **kwargs: kwargs['segment'])),
    ('cpu_usage', ListField('CPU Usage', value_getter=get_usage('cpu'))),
    ('cpu_limit', ListField('CPU Limit', value_getter=get_limit('cpu'))),
    ('mem_usage', ListField('Mem Usage', value_getter=get_usage('mem'), formatter=int_to_str_gigabytes)),
    ('mem_limit', ListField('Mem Limit', value_getter=get_limit('mem'), formatter=int_to_str_gigabytes)),
    ('hdd_usage', ListField('HDD Usage', value_getter=get_disk_usage('hdd'), formatter=int_to_str_gigabytes)),
    ('hdd_limit', ListField('HDD Limit', value_getter=get_disk_limit('hdd'), formatter=int_to_str_gigabytes)),
    ('ssd_usage', ListField('SSD Usage', value_getter=get_disk_usage('ssd'), formatter=int_to_str_gigabytes)),
    ('ssd_limit', ListField('SSD Limit', value_getter=get_disk_limit('ssd'), formatter=int_to_str_gigabytes)),
    ('ipv4_usage', ListField('IPv4 Usage', value_getter=get_usage('internet_address'))),
    ('ipv4_limit', ListField('IPv4 Limit', value_getter=get_limit('internet_address'))),
])

VM_FIELDS = collections.OrderedDict([
    ('name', ListField('Name', attr='meta.id')),
    ('location', ListField('Location', value_getter=lambda obj, **kwargs: kwargs['cluster'])),
    ('cpu_usage', ListField('CPU Usage', attr='spec.qemu.resource_requests.vcpu_guarantee')),
    ('mem_usage', ListField('Mem Usage', attr='spec.qemu.resource_requests.memory_guarantee',
                            formatter=int_to_str_gigabytes)),
    ('hdd_usage', ListField('HDD Usage', value_getter=get_qemu_disk_size('hdd'),
                            formatter=int_to_str_gigabytes)),
    ('ssd_usage', ListField('SSD Usage', value_getter=get_qemu_disk_size('ssd'),
                            formatter=int_to_str_gigabytes)),
    ('ipv4_usage', ListField('IPv4 Usage', attr='spec.qemu.enable_internet', formatter=int))
])


def get_vms_dict(api_client, accounts, cluster):
    vms_dict = {}
    vms = api_client.list_yp_vms(cluster, abc=[x.id for x in accounts])
    for vm in vms:
        account_id = vm.spec.account_id
        segment = vm.spec.qemu.node_segment
        if account_id not in vms_dict:
            vms_dict[account_id] = {}
        if segment not in vms_dict[account_id]:
            vms_dict[account_id][segment] = []
        vms_dict[account_id][segment].append(vm)
    return vms_dict


def run(args):
    api_client = api.VMAgentClient(
        token=args.token,
        proxyhost=args.proxyhost,
        ssl_none=args.ssl_none
    )
    if args.cluster:
        cluster_list = args.cluster
    else:
        cluster_list = api.VMPROXY_LOCATION.keys()
    login = getpass.getuser()

    rows = []
    count = 0
    vms_dict = {}
    accounts_dict = {}
    personal_acc = None
    fields = AVAILABLE_FIELDS.values()
    headers = ['Index'] + [f.field_name for f in fields]
    for cluster in cluster_list:
        accounts = api_client.list_accounts(cluster, login, args.node_segment)
        for acc in accounts:
            accounts_dict['{}#{}'.format(acc.id, cluster)] = acc
        if args.detailed:
            vms_dict[cluster] = get_vms_dict(api_client, accounts, cluster)
    for acc_key in sorted(accounts_dict.keys()):
        acc = accounts_dict[acc_key]
        cluster = acc_key.split('#')[1]
        if acc.type == vmset_pb2.Account.PERSONAL:
            personal_acc = acc
            continue
        for segment in sorted(acc.limits.per_segment.keys()):
            if helpers.is_empty(acc.limits.per_segment[segment]):
                continue
            count += 1
            row = [count]
            for field in fields:
                row.append(field(acc, name=acc.id, cluster=cluster, segment=segment))
            rows.append(row)
            # Detailed account info
            for vm in vms_dict.get(cluster, {}).get(acc.id, {}).get(segment, []):
                row = [None]
                for field_key in AVAILABLE_FIELDS.keys():
                    vm_field = VM_FIELDS.get(field_key)
                    if vm_field is not None:
                        row.append(vm_field(vm, cluster=cluster))
                    else:
                        row.append(None)
                rows.append(row)
    print('Group Quotas')
    print(tabulate(rows, headers=headers, tablefmt='fancy_grid'))

    if personal_acc is not None:
        rows = []
        count = 0
        pa_helper = helpers.PersonalAccountHelper(api_client=api_client, login=login)
        pa_helper.init(accounts=accounts_dict.values())

        # Show personal quota usage
        count += 1
        row = [count]
        personal_usage = pa_helper.to_account()
        for field in fields:
            row.append(field(personal_usage, name='PERSONAL', cluster='', segment=DEV_SEGMENT))
        rows.append(row)

        # Show personal quota vms for detailed view
        for cluster in cluster_list:
            for vm in vms_dict.get(cluster, {}).get(personal_acc.id, {}).get(DEV_SEGMENT, []):
                if vm.meta.author != login:
                    continue
                row = [None]
                for field_key in AVAILABLE_FIELDS.keys():
                    vm_field = VM_FIELDS.get(field_key)
                    if vm_field is not None:
                        row.append(vm_field(vm, cluster=cluster))
                    else:
                        row.append(None)
                rows.append(row)

        # Show normalised group consumption
        count += 1
        row = [count]
        group_usage = pa_helper.group_usage_account()
        for field in fields:
            row.append(field(group_usage, name='Group Cons.', cluster='', segment=DEV_SEGMENT))
        rows.append(row)

        print('Personal Quotas')
        print(tabulate(rows, headers=headers, tablefmt='fancy_grid'))
