import collections
import getpass
import json
import sys

from infra.vmagent.src.vmctl.actions.list_helpers import ListField, int_to_str_gigabytes
from tabulate import tabulate

from infra.vmagent.src.vmctl import api


def list_to_str(value):
    """
    :type value: list
    """
    return ' '.join(value)


def version_str(value):
    """
    Adds v character to make tabulate consider this value as string not as float
    :type value: str
    """
    if value == 'N/A':
        return value
    else:
        return 'v{}'.format(value)


def get_qemu_disk_size(vm, **kwargs):
    """
    :type vm: vmset_pb2.VM
    """
    for vol in vm.spec.qemu.volumes:
        if vol.name == '/qemu-persistent':
            return vol.capacity
    return 0


def host_name_getter(vm, cluster):
    """
    :type vm: vmset_pb2.VM
    :type cluster: str
    """
    name = AVAILABLE_FIELDS['name'](vm, cluster=cluster)
    location = AVAILABLE_FIELDS['location'](vm, cluster=cluster)

    return '{}.{}.yp-c.yandex.net'.format(name, location.lower())


AVAILABLE_FIELDS = collections.OrderedDict([
    ('name', ListField('Name', attr='meta.id')),
    ('location', ListField('Location', attr='meta.cluster',
                           value_getter=lambda vm, cluster: cluster)),
    ('segment', ListField('Segment', attr='spec.qemu.node_segment')),
    ('cpu', ListField('CPU', attr='spec.qemu.resource_requests.vcpu_guarantee')),
    ('ram', ListField('RAM', attr='spec.qemu.resource_requests.memory_guarantee',
                      formatter=int_to_str_gigabytes)),
    ('disk', ListField('Disk',
                       value_getter=get_qemu_disk_size,
                       formatter=int_to_str_gigabytes)),
    ('logins', ListField('Logins', attr='meta.auth.owners.logins',
                         formatter=list_to_str)),
    ('groups', ListField('Groups', attr='meta.auth.owners.group_ids',
                         formatter=list_to_str)),
    ('hostname', ListField('Host', value_getter=host_name_getter, formatter=str)),
    ('abc', ListField('ABC', attr='spec.account_id')),
    ('vmagent_version', ListField('Vmagent', attr='spec.vmagent_version', formatter=version_str))
])


def fetch_yp_vms(args):
    api_client = api.VMAgentClient(
        token=args.token,    # restart.set_defaults(handle=restart_action)
        proxyhost=args.proxyhost,
        ssl_none=args.ssl_none
    )
    # Login argument
    if args.login_all:
        login = None
    else:
        login = args.login or getpass.getuser()
    # Cluster argument
    if args.list_yp_cluster:
        cluster_list = args.list_yp_cluster
    else:
        cluster_list = api.VMPROXY_LOCATION.keys()
    # List field argument
    if args.fields:
        field_set = set(args.fields)
        fields = [f for key, f in AVAILABLE_FIELDS.items() if key in field_set]
    else:
        fields = AVAILABLE_FIELDS.values()
    # Making list
    rows = []
    count = 0
    for cluster in cluster_list:
        try:
            vms = api_client.list_yp_vms(cluster, login, args.node_segment, args.abc)
        except Exception:
            sys.stderr.write("No response from cluster {}\n".format(cluster))
        else:
            for vm in vms:
                count += 1
                row = [count]
                for field in fields:
                    row.append(field(vm, cluster=cluster))
                rows.append(row)

    return rows, ['Index'] + [f.field_name for f in fields]


def list_yp_vms(args):
    rows, headers = fetch_yp_vms(args)

    print(tabulate(rows, headers=headers, tablefmt='fancy_grid'))


def dump_json_yp_vms(args):
    rows, headers = fetch_yp_vms(args)

    print(json.dumps([dict(zip(headers, x)) for x in rows], indent=4, sort_keys=True))


def run(args):
    if args.mode == 'yp':
        if args.format == 'text':
            list_yp_vms(args)
        else:
            dump_json_yp_vms(args)
    elif args.mode == 'gencfg':
        print('Not supported yet')
