#!/usr/bin/python

import base64
import os

import requests

import nanny_rpc_client
from infra.vmagent.src.vmagent_pb import vmagent_api_pb2, vmset_api_pb2, vmset_pb2, vmset_api_stub
from infra.vmagent.src.vmagent_pb import vmagent_pb2
from infra.vmagent.src.vmctl import errors

LOCAL_TOKEN_PATH = '/tmp/vmagent.token'
TIMEOUT = 10
PROXYHOST = 'https://rtc-kvm-vmproxy.n.yandex-team.ru'
VMPROXY_LINK = "{proxyhost}/{path}?host={host}&port={port}&service={service}&pod_id={pod_id}"
DIRECT_LINK = "http://{host}:{port}/{path}?"
DEFAULT_VOLUME_MP = '/qemu-persistent'
VMPROXY_LOCATION = {
    'MAN': 'https://vmproxy.man-swat.yandex-team.ru',
    'SAS': 'https://vmproxy.sas-swat.yandex-team.ru',
    'VLA': 'https://vmproxy.vla-swat.yandex-team.ru',
}
LOCATION_LIST = ['SAS', 'MAN', 'VLA', 'ANY']
DEFAULT_LOCATION = 'SAS'  # for gencfg backup


class VmId(object):
    def __init__(self, cluster, pod_id, host, port, service):
        self.cluster = cluster
        self.pod_id = pod_id
        self.host = host
        self.port = port
        self.service = service

    @classmethod
    def from_arg_parser(cls, args):
        return cls(
            cluster=args.yp_cluster,
            pod_id=args.pod_id,
            host=args.host,
            port=args.port,
            service=args.service
        )


class VMAgentClient(object):
    def __init__(self, host='', port='', service="", token="", pod_id='', yp_cluster='',
                 proxyhost=None, ssl_none=False, timeout=TIMEOUT):
        self.host = host
        self.port = port
        self.service = service
        self.pod_id = pod_id
        self.yp_cluster = yp_cluster.upper() if yp_cluster else None
        self.ssl_verify = not ssl_none
        self.headers = {}

        if service or pod_id:
            link = VMPROXY_LINK
        else:
            link = DIRECT_LINK
            if os.path.exists(LOCAL_TOKEN_PATH):
                with open(LOCAL_TOKEN_PATH) as f:
                    self.headers['Local-Token'] = f.read()

        self.custom_proxyhost = None
        if proxyhost:
            self.custom_proxyhost = proxyhost
        elif pod_id and self.yp_cluster:
            if self.yp_cluster == 'ANY':
                proxyhost = VMPROXY_LOCATION[DEFAULT_LOCATION]
            else:
                proxyhost = VMPROXY_LOCATION[self.yp_cluster]
        else:
            proxyhost = PROXYHOST

        self.status_link = link.format(proxyhost=proxyhost, path="status",
                                       host=host, port=port, service=service, pod_id=self.pod_id)
        self.action_link = link.format(proxyhost=proxyhost, path="action",
                                       host=host, port=port, service=service, pod_id=self.pod_id)
        self.debug_link = link.format(proxyhost=proxyhost, path="debug",
                                      host=host, port=port, service=service, pod_id=self.pod_id)
        self.token = token
        self.headers['Authorization'] = "OAuth {}".format(token)
        self.timeout = timeout

    def get_rpc_stub(self, cluster):
        """
        :type cluster: str
        rtype: vmset_api_stub.VmSetServiceStub
        """
        if self.custom_proxyhost:
            rpc_url = self.custom_proxyhost
        else:
            rpc_url = VMPROXY_LOCATION[cluster.upper()]
        rpc = nanny_rpc_client.RetryingRpcClient(rpc_url + '/api', oauth_token=self.token)
        return vmset_api_stub.VmSetServiceStub(rpc)

    def get_status(self):
        """
        :rtype: vmagent_api_pb2.VMStatusResponse
        """
        resp = requests.get(
            self.status_link,
            headers=self.headers,
            timeout=self.timeout,
            verify=self.ssl_verify
        )
        resp.raise_for_status()
        r = vmagent_api_pb2.VMStatusResponse()
        r.ParseFromString(base64.b64decode(resp.content))
        return r

    def send_action(self, req):
        """
        :type req: google.protobuf.message.Message
        """
        resp = requests.post(
            url=self.action_link,
            data=base64.b64encode(req.SerializeToString()),
            headers=self.headers,
            timeout=self.timeout,
            verify=self.ssl_verify
        )
        resp.raise_for_status()

    def send_debug(self):
        resp = requests.post(
            url=self.debug_link,
            headers=self.headers,
            timeout=self.timeout,
            verify=self.ssl_verify
        )
        resp.raise_for_status()

    def push_config(self, id, vcpu, mem, rb_torrent, disk_size, vm_type, autorun, image_type, raw_req=None):
        """
        :type image_type: str
        :type id: str
        :type vcpu: int
        :type mem: int
        :type rb_torrent: str
        :type disk_size: int
        :type vm_type: str
        :type autorun: bool
        :type raw_req: google.protobuf.message.Message | NoneType
        """
        if raw_req:
            return self.send_action(raw_req)
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.PUSH_CONFIG
        req.config.id = id
        req.config.vcpu = vcpu
        req.config.mem = mem
        req.config.disk.resource.rb_torrent = rb_torrent
        req.config.disk.delta_size = disk_size
        req.config.autorun = autorun
        req.config.disk.type = vmagent_pb2.VMDisk.ImageType.Value(image_type)
        if vm_type.lower() == "linux":
            req.config.type = vmagent_pb2.VMConfig.LINUX
        elif vm_type.lower() == "windows":
            req.config.type = vmagent_pb2.VMConfig.WINDOWS
        else:
            raise errors.VMAgentParamsError('Type {} not supported'.format(vm_type))
        self.send_action(req)

    def start(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.START
        self.send_action(req)

    def rescue(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESCUE
        self.send_action(req)

    def poweroff(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.POWEROFF
        self.send_action(req)

    def reset(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESET
        self.send_action(req)

    def restart(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESTART
        self.send_action(req)

    def shutdown(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.SHUTDOWN
        self.send_action(req)

    def revert(self):
        req = vmagent_api_pb2.VMActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.HARD_RESET
        self.send_action(req)

    def allocate(self, cluster, pod_id, network_id, node_segment, volume_size, mem, vcpu_limit, vcpu_guarantee,
                 storage_class, enable_internet, use_nat64, abc, logins=None, groups=None, layer_url=None):
        """
        :type cluster: str
        :type pod_id: str
        :type network_id: str
        :type node_segment: str
        :type volume_size: int
        :type mem: int
        :type vcpu_limit: int
        :type vcpu_guarantee: int
        :type storage_class: str
        :type enable_internet: bool
        :type use_nat64: bool
        :type abc: str
        :type logins: list[str] | NoneType
        :type groups: list[str] | NoneType
        :type layer_url: str | NoneType
        """
        req = vmset_api_pb2.AllocateVmRequest()
        req.meta.id = pod_id
        if logins:
            req.meta.auth.owners.logins.extend(logins)
        if groups:
            req.meta.auth.owners.group_ids.extend(groups)
        req.spec.type = vmset_pb2.VMSpec.QEMU_VM
        req.spec.qemu.network_id = network_id
        req.spec.qemu.node_segment = node_segment
        req.spec.qemu.resource_requests.dirty_memory_limit = 0
        req.spec.qemu.resource_requests.memory_limit = mem
        req.spec.qemu.resource_requests.anonymous_memory_limit = 0
        req.spec.qemu.resource_requests.vcpu_guarantee = vcpu_guarantee
        req.spec.qemu.resource_requests.vcpu_limit = vcpu_limit
        req.spec.qemu.resource_requests.memory_guarantee = mem
        req.spec.qemu.enable_internet = enable_internet
        req.spec.qemu.use_nat64 = use_nat64
        if abc:
            req.spec.account_id = 'abc:service:{}'.format(abc)
        if layer_url:
            req.spec.qemu.porto_layer.url = layer_url
        v = req.spec.qemu.volumes.add()
        v.name = DEFAULT_VOLUME_MP
        v.capacity = volume_size
        v.storage_class = storage_class
        if cluster == 'ANY':
            self._any_cluster_request(req, self._allocate_request)
        else:
            self._allocate_request(req, cluster)

    def _allocate_request(self, req, cluster):
        """
        :type req: vmset_api_pb2.AllocateVmRequest
        :type cluster: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.allocate_vm(req)
        print('Allocating {} in {}'.format(req.meta.id, cluster))

    def create(self, id, vcpu, mem, rb_torrent, disk_size, vm_type, autorun, cluster, pod_id, network_id,
               node_segment, volume_size, mem_config, vcpu_limit, vcpu_guarantee, storage_class, enable_internet,
               use_nat64, abc, image_type, logins=None, groups=None, layer_url=None, raw_req=None):
        """
        :type image_type: str
        :type cluster: str
        :type pod_id: str
        :type network_id: str
        :type node_segment: str
        :type volume_size: int
        :type mem: int
        :type vcpu_limit: int
        :type vcpu_guarantee: int
        :type storage_class: str
        :type enable_internet: bool
        :type use_nat64: bool
        :type abc: str
        :type logins: list[str] | NoneType
        :type groups: list[str] | NoneType
        :type layer_url: str | NoneType
        :type id: str
        :type vcpu: int
        :type mem_config: int
        :type rb_torrent: str
        :type disk_size: int
        :type vm_type: str
        :type autorun: bool
        :type raw_req: google.protobuf.message.Message | NoneType
        """
        req = vmset_api_pb2.CreateVmRequest()
        req.meta.id = pod_id
        if logins:
            req.meta.auth.owners.logins.extend(logins)
        if groups:
            req.meta.auth.owners.group_ids.extend(groups)
        req.spec.type = vmset_pb2.VMSpec.QEMU_VM
        req.spec.qemu.network_id = network_id
        req.spec.qemu.node_segment = node_segment
        req.spec.qemu.resource_requests.dirty_memory_limit = 0
        req.spec.qemu.resource_requests.memory_limit = mem
        req.spec.qemu.resource_requests.anonymous_memory_limit = 0
        req.spec.qemu.resource_requests.vcpu_guarantee = vcpu_guarantee
        req.spec.qemu.resource_requests.vcpu_limit = vcpu_limit
        req.spec.qemu.resource_requests.memory_guarantee = mem
        req.spec.qemu.enable_internet = enable_internet
        req.spec.qemu.use_nat64 = use_nat64
        if abc:
            req.spec.account_id = 'abc:service:{}'.format(abc)
        if layer_url:
            req.spec.qemu.porto_layer.url = layer_url
        v = req.spec.qemu.volumes.add()
        v.name = DEFAULT_VOLUME_MP
        v.capacity = volume_size
        v.storage_class = storage_class
        if raw_req:
            return self.send_action(raw_req)
        req.config.id = id
        req.config.vcpu = vcpu
        req.config.mem = mem_config
        req.config.disk.resource.rb_torrent = rb_torrent
        req.config.disk.delta_size = disk_size
        req.config.disk.type = vmagent_pb2.VMDisk.ImageType.Value(image_type)
        req.config.autorun = autorun
        if vm_type.lower() == "linux":
            req.config.type = vmagent_pb2.VMConfig.LINUX
        elif vm_type.lower() == "windows":
            req.config.type = vmagent_pb2.VMConfig.WINDOWS
        else:
            raise errors.VMAgentParamsError('Type {} not supported'.format(vm_type))
        if cluster == 'ANY':
            self._any_cluster_request(req, self._create_request)
        else:
            self._create_request(req, cluster)

    def _create_request(self, req, cluster):
        """
        :type req: vmset_api_pb2.CreateVmRequest
        :type cluster: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.create_vm(req)
        print('Creating {} in {}'.format(req.meta.id, cluster))

    def _any_cluster_request(self, req, handler):
        """
        :type req: google.protobuf.message.Message
        :type handler: collections.Callable
        """
        errors_dict = {}
        for cluster_ in VMPROXY_LOCATION.keys():
            try:
                handler(req, cluster_)
            except Exception as e:
                errors_dict[cluster_] = e.message
                continue
            else:
                break
        else:
            errors_str = '\n'.join('{}: {}'.format(key, value) for key, value in errors_dict.items())
            raise errors.VMAgentParamsError("Can't allocate resources in any cluster:\n{}".format(errors_str))

    def deallocate(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        """
        req = vmset_api_pb2.DeallocateVmRequest()
        req.id = pod_id
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.deallocate_vm(req)

    def list_yp_vms(self, cluster, login=None, node_segment=None, abc=None):
        """
        :type cluster: str
        :type login: str | NoneType
        :type node_segment: list[str] | NoneType
        :type abc: list[str] | NoneType
        :rtype collections.Iterable[vmset_pb2.VM]
        """
        req = vmset_api_pb2.ListYpVmRequest()
        if login:
            req.query.login = login
        if node_segment:
            req.query.segment.extend(node_segment)
        if abc:
            req.query.account.extend(abc)
        rpc_stub = self.get_rpc_stub(cluster)
        rsp = rpc_stub.list_yp_vm(req)
        return rsp.vms

    def update(self, cluster, pod_id, abc, update_vmagent, logins=None, groups=None, clear_groups=False):
        """
        :type cluster: str
        :type pod_id: str
        :type abc: str | None
        :type update_vmagent: bool
        :type logins: list[str]
        :type groups: list[str]
        :type clear_groups: bool
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.GetVmRequest()
        req.vm_id = pod_id
        vm = rpc_stub.get_vm(req).vm
        if abc:
            vm.spec.account_id = 'abc:service:{}'.format(abc)

        req = vmset_api_pb2.UpdateVmRequest()
        req.update_vmagent = update_vmagent
        req.meta.CopyFrom(vm.meta)
        req.spec.CopyFrom(vm.spec)

        if logins:
            req.meta.auth.owners.ClearField("logins")
            req.meta.auth.owners.logins.extend(sorted(list(set(logins))))

        if groups:
            req.meta.auth.owners.ClearField("group_ids")
            req.meta.auth.owners.group_ids.extend(sorted(list(set(groups))))

        if clear_groups:
            req.meta.auth.owners.ClearField("group_ids")

        return rpc_stub.update_vm(req)

    def backup(self, vm_id):
        req = vmset_api_pb2.BackupVmRequest()
        if vm_id.cluster and vm_id.pod_id:
            req.vm_id.pod_id = vm_id.pod_id
            rpc_stub = self.get_rpc_stub(vm_id.cluster)
        else:
            req.vm_id.nanny_args.host = vm_id.host
            req.vm_id.nanny_args.port = vm_id.port
            req.vm_id.nanny_args.service = vm_id.service
            rpc_stub = self.get_rpc_stub(DEFAULT_LOCATION)
        rpc_stub.backup_vm(req)

    def get_vm(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        :rtype: vmset_pb2.VM
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.GetVmRequest()
        req.vm_id = pod_id
        return rpc_stub.get_vm(req).vm

    def list_backups(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        :rtype: collections.Iterable[vmset_pb2.Backup]
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.ListBackupRequest()
        req.vm_id = pod_id
        return rpc_stub.list_backup(req).backups

    def remove_backup(self, cluster, pod_id, backup_id):
        """
        :type cluster: str
        :type pod_id: str
        :type backup_id: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.RemoveBackupRequest()
        req.vm_id = pod_id
        req.id = backup_id
        rpc_stub.remove_backup(req)

    def list_accounts(self, cluster, login, node_segment=None):
        """
        :type cluster: str
        :type login: str
        :type node_segment: str | NoneType
        :rtype: list
        """
        req = vmset_api_pb2.ListUserAccountsRequest()
        req.login = login
        if node_segment:
            req.segment = node_segment
        return self.get_rpc_stub(cluster).list_user_accounts(req).accounts
