#!/usr/bin/python

import argparse
# import codecs
import getpass
# import locale
import os
import re
import six
import sys
import time
import urllib
import uuid
import webbrowser
from distutils.version import LooseVersion

import requests
import requests.exceptions

import library.python.oauth as lpo
import nanny_rpc_client
from google.protobuf import json_format
from infra.vmagent.src.vmagent_pb import vmagent_pb2, vmset_pb2
from infra.vmagent.src.vmctl import errors, helpers
from infra.vmagent.src.vmctl.actions import list_accounts
from infra.vmagent.src.vmctl.actions import list_action
from infra.vmagent.src.vmctl.actions import list_backup
from library.python.svn_version import svn_revision
from . import api

# sys_encoding = locale.getpreferredencoding()
# sys.stdin = codecs.getwriter(sys_encoding)(sys.stdin)
# sys.stdout = codecs.getwriter(sys_encoding)(sys.stdout)
# sys.stderr = codecs.getwriter(sys_encoding)(sys.stderr)

VMCTL_LINUX_URL = 'https://proxy.sandbox.yandex-team.ru/last/VMCTL_LINUX'
VMCTL_MACOS_URL = 'https://proxy.sandbox.yandex-team.ru/last/VMCTL_MACOS'
VNC_LINK = 'https://rtc-kvm-novnc.yandex-team.ru/vnc.html?path=?{path}'
VNC_OFFSET = 3
DISTRO_DICT = {
    "xenial": "rbtorrent:6e7bdce06445ddf136401ffbd4071e0e36003579",
    "trusty": "rbtorrent:7cf577a3a5a54106fd4d1ad7e70c9011a9cc9f9c",
    "precise": "rbtorrent:c79c6711f1392e77f1ab76d2f8c7f8afbf01d483",
    "bionic": "rbtorrent:da9bed3db4df29784d5fe8d3f16b9f6ef6819ddd",
    "xenial-dev": "rbtorrent:6942c5cbb9eb91e4841c0c54d5b1547889abb063",
    "trusty-dev": "rbtorrent:a21c8d4174a53be6ac3ec42c2da8e8d84914ab74",
    "precise-dev": "rbtorrent:a3a282bc44a8cb4c7d5fbfb3ac9e83a8363cd403",
    "bionic-dev": "rbtorrent:bf20b3c2d050d41182d90e3754288c0bb28b4a1e"
}

CLIENT_ID = '197597bbc8e3426187209d78f1d75e8b'
CLIENT_SECRET = 'e24a619bb6d0402cb54356d685be5d3a'

ANSWERS_YES = (
    'yes',
    'y',
)
ANSWERS_NO = (
    'no',
    'n',
)
ANSWERS = ANSWERS_YES + ANSWERS_NO

TMP_ACC_WARN_PROMPT = """You are going to use temporary account.
Please, keep in mind, that your VM could be deallocated automatically at any time!
Continue? [Y/N]:"""
VM_POWEROFF_PROMPT = """VM will be stopped during this action, all unsaved data will be lost.
Continue? [Y/N]:"""
VMAGENT_UPDATE_PROMPT = """Used vmagent version does not support this operation.
Would you like to update vmagent to latest version? [Y/N]:"""
VM_BACKUP_FAIL_PROMPT = """Backup action are in progress now. It will fail after VM start.
Continue? [Y/N]"""
OBJECT_REMOVE_PROMPT_TMPL = """{} will be removed.
Continue? [Y/N]"""
MACRO_PATTERN = re.compile("^_[A-Z0-9_]+_$")
WAIT_ITERATION_TIMEOUT = 10
FINITE_STATES = frozenset([
    vmagent_pb2.VMState.CONFIGURED,
    vmagent_pb2.VMState.STOPPED,
    vmagent_pb2.VMState.RUNNING
])
VMAGENT_BACKUP_VERSION = LooseVersion('0.15')
PERSONAL_ACCOUNT = 'abc:service:4172'


def ApiClientFromArgs(args):
    args_dict = vars(args)
    return api.VMAgentClient(
        host=args_dict.get("host", ''),
        port=args_dict.get("port", ''),
        pod_id=args_dict.get("pod_id", ''),
        yp_cluster=args_dict.get("yp_cluster", ''),
        service=args_dict.get("service", ''),
        token=args_dict.get("token", ''), proxyhost=args_dict.get("proxyhost", None),
        ssl_none=args_dict.get("ssl_none", False), timeout=args_dict.get("timeout", api.TIMEOUT),
    )


def wait_status_change(client, args):
    while True:
        try:
            status = client.get_status()
        except requests.exceptions.HTTPError as e:
            if args.verbose:
                print(e.response.content or e)
        else:
            if status.state.type in FINITE_STATES:
                return status
            if args.verbose:
                print(status.state)
        time.sleep(WAIT_ITERATION_TIMEOUT)


def check_qyp_vmagent(client, args):
    vm = client.get_vm(args.yp_cluster, args.pod_id)
    if vm.spec.vmagent_version == 'N/A':
        used_version = LooseVersion('0')
    else:
        used_version = LooseVersion(vm.spec.vmagent_version)
    if used_version < VMAGENT_BACKUP_VERSION:
        if not check_with_custom_prompt(args, VMAGENT_UPDATE_PROMPT):
            print('Interrupted by user')
            sys.exit()
        client.update(args.yp_cluster, args.pod_id, None, True)
        wait_status_change(client, args)


def self_upgrade_action(_):
    if sys.platform == "linux" or sys.platform == "linux2":
        url = VMCTL_LINUX_URL
    elif sys.platform == "darwin":
        url = VMCTL_MACOS_URL
    else:
        # FIXME: implement self-upgrade for win32
        print('upgrade-vmctl is not yet supported for %s' % sys.platform)
        sys.exit(1)
    me = os.path.abspath(sys.argv[0])
    # If we are executed from Arcadia PY_PROGRAM binary container
    # sys.argv[0] path is incorrect, we should sys.executable instead
    me_exe = os.path.abspath(sys.executable)
    if os.path.basename(me) == os.path.basename(me_exe):
        me = me_exe
    me_tmp = me + '.tmp'
    print("downloading " + url + " -> " + me)
    r = requests.get(url)
    with open(me_tmp, 'wb') as f:
        for chunk in r.iter_content(chunk_size=65536):
            if chunk:  # filter out keep-alive new chunks
                f.write(chunk)

    os.chmod(me_tmp, 0o755)
    os.rename(me_tmp, me)
    print("upgrade successful")


def get_status_action(args):
    m = ApiClientFromArgs(args).get_status()
    print(json_format.MessageToJson(m, including_default_value_fields=True))


def monitor_action(args):
    c = ApiClientFromArgs(args)

    while True:
        print(c.get_status())
        time.sleep(1)


def push_config_action(args):
    c = ApiClientFromArgs(args)

    if not args.type:
        print("No VM type specified, considering Linux")
        args.type = "linux"

    c.push_config(id=str(uuid.uuid4()),
                  vcpu=args.vcpu,
                  mem=args.mem,
                  rb_torrent=args.rb_torrent,
                  disk_size=args.disk_size,
                  vm_type=args.type,
                  image_type=args.image_type,
                  autorun=args.autorun)


def start_action(args):
    client = ApiClientFromArgs(args)
    if args.yp_cluster and args.pod_id:
        backups = client.list_backups(args.yp_cluster, args.pod_id)
        for backup in backups:
            if backup.status.state in (vmset_pb2.BackupStatus.PLANNED, vmset_pb2.BackupStatus.IN_PROGRESS):
                if not check_with_custom_prompt(args, VM_BACKUP_FAIL_PROMPT):
                    print('Interrupted by user')
                    sys.exit()
    client.start()


def rescue_action(args):
    ApiClientFromArgs(args).rescue()


def poweroff_action(args):
    ApiClientFromArgs(args).poweroff()


def reset_action(args):
    ApiClientFromArgs(args).reset()


def restart_action(args):
    ApiClientFromArgs(args).restart()


def shutdown_action(args):
    ApiClientFromArgs(args).shutdown()


def revert_action(args):
    ApiClientFromArgs(args).revert()


def backup_action(args):
    if not check_with_custom_prompt(args, VM_POWEROFF_PROMPT):
        print('Interrupted by user')
        sys.exit()

    client = ApiClientFromArgs(args)
    if args.yp_cluster and args.pod_id:
        check_qyp_vmagent(client, args)
    vm_id = api.VmId.from_arg_parser(args)
    client.backup(vm_id)


def vnc_action(args):
    rsp = ApiClientFromArgs(args).get_status()

    if rsp.state.type != vmagent_pb2.VMState.RUNNING:
        print("VM is not running, VNC is not accessible now")
        return

    if args.pod_id and args.yp_cluster:
        query = {
            'pod_id': args.pod_id,
            'cluster': args.yp_cluster
        }
    else:
        query = {
            'host': os.environ.get('PORTO_HOST') or args.host,
            'port': args.port,
            'service': args.service
        }

    link = VNC_LINK.format(path=urllib.quote(urllib.urlencode(query)))
    pwd = rsp.config.access_info.vnc_password
    if pwd:
        link += "&password={}".format(pwd)

    print("VNC url: {}".format(link))
    print("VNC password: {}".format(rsp.config.access_info.vnc_password))

    webbrowser.open_new(link)


def noempty_str(value):
    if value is None or not value:
        raise argparse.ArgumentTypeError('Non empty value required')
    return value


def check_network_id(args):
    subparser = all_subparsers[vars(args)["short_name"]]
    if not args.network_id:
        args.network_id = os.environ.get('VMCTL_NETWORK_ID', '')
        if not args.network_id and args.node_segment == "dev":
            args.network_id = "_SEARCHSAND_"
    value = args.network_id
    if value is None or not value:
        subparser.error('argument --network-id: Non empty value required')
    if not MACRO_PATTERN.match(value.lstrip()):
        msg = 'Invalid racktables macro \"{}\", should match the pattern: _[A-Z0-9_]+_'.format(value)
        subparser.error(msg)


def allocate_action(args):
    c = ApiClientFromArgs(args)
    storage_class = args.storage_class or {'dev': 'ssd'}.get(args.node_segment, 'hdd')
    if args.abc == 'personal':
        pa_helper = helpers.PersonalAccountHelper(api_client=c, login=getpass.getuser())
        pa_helper.init()
        pa_helper.validate_new_resource_fit(cpu=args.vcpu_guarantee,
                                            mem=args.memory,
                                            disk_size=args.volume_size,
                                            storage_class=storage_class,
                                            enable_internet=args.enable_internet)
        abc = PERSONAL_ACCOUNT.split('abc:service:')[1]
    else:
        abc = args.abc

    c.allocate(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        network_id=args.network_id,
        node_segment=args.node_segment,
        volume_size=args.volume_size,
        mem=args.memory,
        vcpu_limit=args.vcpu_limit,
        vcpu_guarantee=args.vcpu_guarantee,
        logins=args.logins,
        groups=args.groups,
        layer_url=args.layer_url,
        storage_class=storage_class,
        enable_internet=args.enable_internet,
        use_nat64=args.use_nat64,
        abc=abc,
    )


def create_action(args):
    c = ApiClientFromArgs(args)
    if not args.type:
        print("No VM type specified, considering Linux")
        args.type = "linux"

    storage_class = args.storage_class or {'dev': 'ssd'}.get(args.node_segment, 'hdd')
    if args.abc == 'personal':
        pa_helper = helpers.PersonalAccountHelper(api_client=c, login=getpass.getuser())
        pa_helper.init()
        pa_helper.validate_new_resource_fit(cpu=args.vcpu_guarantee * 1000,
                                            mem=args.memory,
                                            disk_size=args.volume_size,
                                            storage_class=storage_class,
                                            enable_internet=args.enable_internet)
        abc = PERSONAL_ACCOUNT.split('abc:service:')[1]
    else:
        abc = args.abc
    c.create(
        id=str(uuid.uuid4()),
        vcpu=args.vcpu,
        mem_config=args.memory - 1024 ** 3,
        rb_torrent=args.rb_torrent,
        disk_size=0,
        vm_type=args.type,
        image_type=args.image_type,
        autorun=True,

        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        network_id=args.network_id,
        node_segment=args.node_segment,
        volume_size=args.volume_size,
        mem=args.memory,
        vcpu_limit=args.vcpu_limit * 1000,
        vcpu_guarantee=args.vcpu_guarantee * 1000,
        logins=args.logins,
        groups=args.groups,
        layer_url=args.layer_url,
        storage_class=storage_class,
        enable_internet=args.enable_internet,
        use_nat64=args.use_nat64,
        abc=abc,
    )


def deallocate_action(args):
    ApiClientFromArgs(args).deallocate(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
    )


def hostname_action(args):
    if args.pod_id and args.yp_cluster:
        print('{}.{}.yp-c.yandex.net'.format(args.pod_id, args.yp_cluster.lower()))
    elif args.host:
        print(args.host)


def update_action(args):
    if args.update_vmagent:
        if not check_with_custom_prompt(args, VM_POWEROFF_PROMPT):
            print('Interrupted by user')
            sys.exit()

    resp = ApiClientFromArgs(args).update(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        abc=args.abc,
        update_vmagent=args.update_vmagent,
        logins=args.logins,
        groups=args.groups,
        clear_groups=args.clear_groups
    )

    if args.logins or args.groups or args.clear_groups:
        print("Owners success updated: ")
        print(resp.vm.meta.auth.owners)


def remove_backup_action(args):
    if not check_with_custom_prompt(args, OBJECT_REMOVE_PROMPT_TMPL.format('Backup')):
        print('Interrupted by user')
        sys.exit()
    ApiClientFromArgs(args).remove_backup(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        backup_id=args.id,
    )


def pod_id_str(value, pattern=re.compile(r"^[a-z0-9\-]+$")):
    if value and not pattern.match(value.lstrip()):
        raise argparse.ArgumentTypeError("Invalid pod id \"{}\", should match the pattern: [a-z0-9\\-]+".format(value))
    return value


def size_suffix_str(value, pattern=re.compile(r'^([0-9]+)([KkMmGgTt]?)$')):
    if not value:
        return value

    res = pattern.match(value)

    if not res:
        msg = 'Invalid size value \"{}\", valid suffixes are Kk, Mm, Gg and Tt or none'.format(value)
        raise argparse.ArgumentTypeError(msg)

    num_str, suffix = res.groups()

    numeric = int(num_str)

    if suffix.lower() == 'k':
        numeric *= 1024

    elif suffix.lower() == "m":
        numeric *= 1048576

    elif suffix.lower() == "g":
        numeric *= 1048576 * 1024

    elif suffix.lower() == "t":
        numeric *= 1048576 * 1048576

    return numeric


def guess_proper_yp_cluster(args):
    def iter_clusters():
        for cluster in api.VMPROXY_LOCATION.keys():
            vms = ApiClientFromArgs(args).list_yp_vms(cluster, login=getpass.getuser())
            for x in vms:
                if x.meta.id == args.pod_id:
                    yield cluster

    clusters = list(frozenset(iter_clusters()))
    if len(clusters) == 1:
        args.yp_cluster = clusters[0]


all_subparsers = {}


def correctness_of_vm_address(args):
    # check for YP
    subparser = all_subparsers[vars(args)["short_name"]]
    if args.pod_id or args.yp_cluster:
        if args.service or args.host or args.port:
            subparser.error("incorrect address for VMProxy (connection with YP): "
                            "port or host or service specified")
        if args.pod_id and not args.yp_cluster:
            guess_proper_yp_cluster(args)
        if not args.yp_cluster or not args.pod_id:
            subparser.error("incorrect address for VMProxy (connection with YP): "
                            "no pod_id or cluster specified")
    # check for VMAgent
    elif args.direct:
        if args.pod_id or args.yp_cluster or args.service:
            subparser.error("incorrect address for VMAgent: pod_id or cluster "
                            "or service specified")
    # check for gencfg/nanny
    elif not args.host or not args.port or not args.service:
        subparser.error("incorrect address for VMProxy (connection with gencfg/nanny): "
                        "no host or port or service specified")


old_style_commands = frozenset(["status", "monitor", "config", "start", "shutdown",
                                "poweroff", "reset", "revert", "rescue", "hostname", "vnc"])
destructive_commands = frozenset(['config', 'deallocate', 'revert'])


def init_argparse():
    confirm_parser = argparse.ArgumentParser(add_help=False)
    confirm_parser.add_argument('-y', dest='assume_yes', action="store_true", help="Automatic yes to prompts")

    pod_parser = argparse.ArgumentParser(add_help=False)
    pod_parser.add_argument('--cluster', dest='yp_cluster', type=noempty_str,
                            choices=api.VMPROXY_LOCATION.keys(),
                            default=os.environ.get('VMCTL_CLUSTER', ''))
    pod_parser.add_argument("--pod-id", dest="pod_id", help="pod_id", type=pod_id_str, required=True)

    action_parser = argparse.ArgumentParser(add_help=False)
    action_parser.add_argument("-H", "--host", dest="host", type=str, help="Instance host")
    action_parser.add_argument("-P", "--port", dest="port", type=int, help="Instance port")
    action_parser.add_argument("-S", "--service", dest="service", type=str, help="Service name", default='')
    action_parser.add_argument('--cluster', dest='yp_cluster', type=str,
                               choices=api.VMPROXY_LOCATION.keys(),
                               default=os.environ.get('VMCTL_CLUSTER', ''))
    action_parser.add_argument("--pod-id", dest="pod_id", help="pod_id", type=pod_id_str, default='')
    action_parser.add_argument("--direct", dest="direct", action="store_true", default=False)
    action_parser.add_argument("--timeout", dest="timeout", type=int, help="Connect timeout", default=None)

    parser = argparse.ArgumentParser()
    parser.add_argument("-T", "--token", dest="token", type=str,
                        help="OAuth token (default from VMCTL_TOKEN environment variable)", default='')
    parser.add_argument("-k", "--no-check-certificate", dest="ssl_none",
                        help="Do not verify proxy host SSL certificate", action='store_true')
    parser.add_argument("-L", "--server", dest="proxyhost", type=str, help="VMProxy hostname")
    parser.add_argument('-V', '--verbose', action="store_true", help="Verbose output")
    parser.add_argument('--version', action='version', help='show version info',
                        version='vmctl svn revision: r{}'.format(svn_revision()))

    subparsers = parser.add_subparsers(title="Possible actions")
    if 'ya' not in os.path.basename(sys.executable):
        upgrade = subparsers.add_parser(name="upgrade-vmctl", description='upgrade vmctl binary to most recent version')
        upgrade.set_defaults(handle=self_upgrade_action, short_name="upgrade-vmctl")

    status = subparsers.add_parser(name="status", description='Retrieve vm instance status', parents=[action_parser])
    status.set_defaults(handle=get_status_action, short_name="status")

    monitor = subparsers.add_parser(name="monitor", description='Monitor instance status', parents=[action_parser])
    monitor.set_defaults(handle=monitor_action, short_name="monitor")

    base_config_parser = argparse.ArgumentParser(add_help=False)
    os_choice = base_config_parser.add_mutually_exclusive_group(required=True)
    os_choice.add_argument("--default-image", dest="d_image", type=str, choices=DISTRO_DICT.keys(),
                           help="choose one of prebuilt ubuntu image")
    os_choice.add_argument("-i", "--image", dest="rb_torrent", type=str, help="VM Image rbtorrent")
    base_config_parser.add_argument("-c", "--vcpu", dest="vcpu", type=int, help="VM vcpu count", required=True)
    base_config_parser.add_argument("-t", "--type", dest="type", type=str, help="VM type")
    base_config_parser.add_argument("--image-type", choices=["RAW", "DELTA"], default="DELTA")

    config = subparsers.add_parser(name="config", description='Set or replace instance vm configuration',
                                   parents=[action_parser, confirm_parser, base_config_parser])
    config.set_defaults(handle=push_config_action, short_name="config")
    config.add_argument("-m", "--mem", dest="mem", type=size_suffix_str, required=True,
                        help="Desired memory size in bytes")
    config.add_argument("--autorun", dest='autorun', default=False, action='store_true')
    config.add_argument("-s", "--disk", dest="disk_size", type=size_suffix_str,
                        help="Desired disk size in bytes", default=0)

    start = subparsers.add_parser(name="start", description='Start vm in instance',
                                  parents=[action_parser, confirm_parser])
    start.set_defaults(handle=start_action, short_name="start")

    rescue = subparsers.add_parser(name="rescue", description='Launch rescue shell with instance disk',
                                   parents=[action_parser])
    rescue.set_defaults(handle=rescue_action, short_name="rescue")

    poweroff = subparsers.add_parser(name="poweroff", description='Stop vm in instance', parents=[action_parser])
    poweroff.set_defaults(handle=poweroff_action, short_name="poweroff")

    reset = subparsers.add_parser(name="reset", description='Reset vm in instance', parents=[action_parser])
    reset.set_defaults(handle=reset_action, short_name="reset")

    # restart = subparsers.add_parser(name="restart",
    #                                 description='Try to gracefully restart vm in instance')
    # restart.set_defaults(handle=restart_action)

    shutdown = subparsers.add_parser(name="shutdown", description='Try to gracefully shutdown vm in instance',
                                     parents=[action_parser])
    shutdown.set_defaults(handle=shutdown_action, short_name="shutdown")

    revert = subparsers.add_parser(name="revert", description='Revert vm to its initial state',
                                   parents=[confirm_parser, action_parser])
    revert.set_defaults(handle=revert_action, short_name="revert")

    backup = subparsers.add_parser(name='backup', description='Backup vm disk',
                                   parents=[action_parser, confirm_parser])
    backup.set_defaults(handle=backup_action, short_name='backup')

    vnc = subparsers.add_parser(name="vnc", description='Open link to instance vnc in external browser',
                                parents=[action_parser])
    vnc.set_defaults(handle=vnc_action, short_name="vnc")

    base_allocate_parser = argparse.ArgumentParser(add_help=False)
    base_allocate_parser.add_argument('--cluster', dest='yp_cluster', type=noempty_str,
                                      choices=api.LOCATION_LIST,
                                      default=os.environ.get('VMCTL_CLUSTER', ''))
    base_allocate_parser.add_argument("--pod-id", dest="pod_id", help="pod_id", type=pod_id_str, required=True)
    base_allocate_parser.add_argument('--network-id', dest='network_id',
                                      help=" (default from VMCTL_NETWORK_ID environment variable)")
    base_allocate_parser.add_argument('--node-segment', dest='node_segment', type=noempty_str,
                                      help=" (default from VMCTL_NODE_SEGMENT environment variable)",
                                      choices=["dev", "default"],
                                      default=os.environ.get('VMCTL_NODE_SEGMENT', ''))
    base_allocate_parser.add_argument('--volume-size', dest='volume_size', type=size_suffix_str, default='20G')
    base_allocate_parser.add_argument('--memory', dest='memory', type=size_suffix_str, default=8589934592)
    base_allocate_parser.add_argument('--logins', nargs='+', dest='logins', type=noempty_str,
                                      help=" (default from VMCTL_LOGINS environment variable, or from current user: %(default)s)",
                                      default=os.environ.get('VMCTL_LOGINS', [getpass.getuser()]))
    base_allocate_parser.add_argument('--groups', nargs='+', dest='groups', type=str)
    base_allocate_parser.add_argument('--layer-url', dest='layer_url', type=str, help=argparse.SUPPRESS)
    base_allocate_parser.add_argument('--storage', dest='storage_class', type=str, default='')
    base_allocate_parser.add_argument('--use-white-ipv4', dest='enable_internet', default=False, action='store_true')
    base_allocate_parser.add_argument('--use-nat64', dest='use_nat64', default=False, action='store_true')
    base_allocate_parser.add_argument('--abc', dest='abc', type=str)

    allocate = subparsers.add_parser(name='allocate', description='Allocate pod for vm in YP',
                                     parents=[base_allocate_parser, confirm_parser])
    allocate.set_defaults(handle=allocate_action, short_name="allocate")
    allocate.add_argument('--vcpu-limit', dest='vcpu_limit', type=int, default=32000)
    allocate.add_argument('--vcpu-guarantee', dest='vcpu_guarantee', type=int, default=2000)

    create = subparsers.add_parser(name='create',
                                   description='Allocate pod for vm in YP and set or replace instance vm configuration',
                                   parents=[base_config_parser, base_allocate_parser, confirm_parser])
    create.set_defaults(handle=create_action, short_name="create")
    create.add_argument('--vcpu-limit', dest='vcpu_limit', type=int, default=32)
    create.add_argument('--vcpu-guarantee', dest='vcpu_guarantee', type=int, default=2)

    deallocate = subparsers.add_parser(name='deallocate', description='Deallocate pod for vm in YP',
                                       parents=[pod_parser, confirm_parser])
    deallocate.set_defaults(handle=deallocate_action, short_name="deallocate")

    hostname = subparsers.add_parser(name='hostname', description='Show VM hostname', parents=[action_parser])
    hostname.set_defaults(handle=hostname_action, short_name="hostname")

    list_cmd = subparsers.add_parser(name='list', description='Show list vms')
    list_cmd.set_defaults(handle=list_action.run, short_name="list")
    list_cmd.add_argument('--mode', dest='mode', type=str, default='yp', choices=('gencfg', 'yp'))
    list_cmd.add_argument('--format', dest='format', type=str, default='text', choices=('text', 'json'))
    list_cmd.add_argument('--cluster', nargs='+', dest='list_yp_cluster', type=str,
                          choices=api.VMPROXY_LOCATION.keys())
    list_cmd.add_argument('--login', dest='login', type=str)
    list_cmd.add_argument('--login-all', dest='login_all', default=False, action='store_true')
    list_cmd.add_argument('--node-segment', nargs='+', dest='node_segment', type=str)
    list_cmd.add_argument('--fields', nargs='+', dest='fields', type=str,
                          choices=list_action.AVAILABLE_FIELDS.keys())
    list_cmd.add_argument('--abc', nargs='+', dest='abc', type=str)
    update = subparsers.add_parser(name='update', description='Update vm',
                                   parents=[pod_parser, confirm_parser])
    update.set_defaults(handle=update_action, short_name='update')
    update.add_argument('--abc', dest='abc', type=str, required=False)
    update.add_argument('--update-vmagent', dest='update_vmagent', default=False, action='store_true')
    update.add_argument('--logins', nargs='+', help="Update logins. If not pass, has no effect", dest='logins', type=str)
    update.add_argument('--groups', nargs='+', help="Update groups. If not pass, has no effect", dest='groups', type=str)
    update.add_argument('--clear-groups', dest='clear_groups', help="Clear all groups from owners",
                        default=False, action='store_true')

    list_backup_cmd = subparsers.add_parser(name='list-backup',
                                            description='Show list vm backups',
                                            parents=[pod_parser])
    list_backup_cmd.set_defaults(handle=list_backup.run, short_name='list-backup')

    remove_backup = subparsers.add_parser(name='remove-backup',
                                          description='Remove vm backup',
                                          parents=[pod_parser, confirm_parser])
    remove_backup.set_defaults(handle=remove_backup_action, short_name='remove-backup')
    remove_backup.add_argument('--id', dest='id', type=str, required=True)

    list_accounts_cmd = subparsers.add_parser(name='list-accounts', description='Show list YP accounts')
    list_accounts_cmd.set_defaults(handle=list_accounts.run, short_name='list-accounts')
    list_accounts_cmd.add_argument('--cluster', nargs='+', dest='cluster', type=str,
                                   choices=api.VMPROXY_LOCATION.keys())
    list_accounts_cmd.add_argument('--segment', dest='node_segment', type=str,
                                   choices=['dev', 'default'])
    list_accounts_cmd.add_argument('--detailed', dest='detailed', default=False, action='store_true')
    global all_subparsers
    all_subparsers = subparsers._name_parser_map.copy()
    return parser


def confirm(prompt):
    while True:
        answer = six.moves.input(prompt).lower()
        if answer not in ANSWERS:
            message = "Incorrect answer. Should be one of: {}".format(', '.join(ANSWERS))
            print(message)
            continue
        if answer in ANSWERS_YES:
            return True
        return False


def check_with_custom_prompt(args, prompt):
    if args.assume_yes:
        return True
    return confirm(prompt)


def check_for_delete_action(args):
    if vars(args)['assume_yes']:
        return True

    # FIXME: uncomment when fix error on get_status
    # state = ApiClientFromArgs(args).get_status().state.type
    # if state == vmagent_pb2.VMState.EMPTY:
    #     return True
    return confirm("All data in selected virtual machine will be deleted [Y/N]:")


def main(cmdline=None):
    if cmdline is None:
        cmdline = sys.argv[1:]

    parser = init_argparse()
    args = parser.parse_args(cmdline)
    subparser = all_subparsers[vars(args)["short_name"]]
    args.token = args.token or os.environ.get('VMCTL_TOKEN', '') or lpo.get_token(CLIENT_ID, CLIENT_SECRET)

    if not args.token:
        parser.error('argument -T/--token: Non empty value required')

    if vars(args)["short_name"] in old_style_commands:
        correctness_of_vm_address(args)

        # defaults for VMAgent
        if args.direct:
            if not args.host:
                args.host = "[::1]"
            if not args.port:
                args.port = os.environ.get('SERVICE_PORT', 7255)

    if vars(args)["short_name"] == "config" or vars(args)["short_name"] == "create":
        if args.d_image and args.type:
            subparser.error("you cannot choose type of OS (windows or linux) if you chose image from ready images")
        if args.rb_torrent and not args.type:
            subparser.error("choose type of OS (windows or linux)")

        if args.d_image:
            args.type = "linux"
            args.rb_torrent = DISTRO_DICT[args.d_image]

    if vars(args)["short_name"] in destructive_commands:
        if not check_for_delete_action(args):
            print("Interrupted by user")
            sys.exit()

    if args.short_name in ('allocate', 'create') and not args.abc:
        if not check_with_custom_prompt(args, TMP_ACC_WARN_PROMPT):
            print('Interrupted by user')
            sys.exit()

    if 'network_id' in vars(args):
        check_network_id(args)

    if args.verbose:
        args.handle(args)
    else:
        try:
            args.handle(args)
        except errors.VMAgentParamsError as e:
            print(e.message or e)
        except requests.HTTPError as e:
            print(e.response.content or e)
        except requests.ConnectionError as e:
            print('Cannot connect to vmproxy server: {}'.format(e))
        except nanny_rpc_client.exceptions.RpcError as e:
            print(e)
        except nanny_rpc_client.exceptions.BalancerRetriesExceeded:
            print('Request failed. Try again later')


if __name__ == '__main__':
    main()
