package kexeclib

import (
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"strconv"
)

type Job int

const (
	REBOOT  Job = 1
	UPGRADE Job = 2
)

type FlagSet struct {
	Job Job
	Dry bool
}

//cmdline flags parser function
func GetFlags() (FlagSet, error) {
	flags := FlagSet{}
	dryPtr := flag.Bool("dry", false, "dry mode, print raw kexec commandline and exit")
	rebootPtr := flag.Bool("reboot", false, "kexec reboot instead any of need_kernel_reboot value")
	upgradePtr := flag.Bool("upgrade", false, "kexec reboot only for kernel upgrade")
	flag.Parse()
	if *dryPtr {
		flags.Dry = true
	}
	if *upgradePtr {
		flags.Job = UPGRADE
	}
	if *rebootPtr {
		flags.Job = REBOOT
	}
	if flag.NFlag() == 0 {
		err := errors.New("you should use one of commandline flags: --dry, --reboot or --upgrade")
		return flags, err
	}
	if *rebootPtr && *upgradePtr {
		//ToDo find less sophisticated solution for incompatible flags
		err := errors.New("combination of incompatible flags --reboot and --upgrade set, chase one of")
		return flags, err
	}
	return flags, nil
}

type KernelMessage struct {
	//ya-salt kernel section state structure
	NeedReboot    bool
	KernelVersion string
}

type yaSaltMessage struct {
	//ya-salt high level parser
	Kernel kernMessage `json:"kernel"`
}

type kernMessage struct {
	//kernel section parser
	NeedRebootMessage  needRebootMessage  `json:"needReboot"`
	BootVersionMessage bootVersionMessage `json:"bootVersion"`
}

type needRebootMessage struct {
	//need kernel reboot section parser
	Status string `json:"status"`
}

type bootVersionMessage struct {
	//current kernel version parser
	Version string `json:"version"`
}

//Ya-salt kernel message parser
func GetKernelVersion(yasaltOutput string) (*KernelMessage, error) {
	ko := &KernelMessage{}
	km := yaSaltMessage{}
	err := json.Unmarshal([]byte(yasaltOutput), &km)
	if err != nil {
		return nil, err
	}
	kernelVersion := km.Kernel.BootVersionMessage.Version
	needReboot := km.Kernel.NeedRebootMessage.Status
	if kernelVersion == "" || needReboot == "" {
		err = errors.New("kernel version or rebootFlag doesnt set in yasalt")
		return nil, err
	}
	needRebootFlag, err := strconv.ParseBool(needReboot)
	if err != nil {
		println("not a bool value for needReboot in ya-salt json")
		return nil, err
	}
	ko.KernelVersion, ko.NeedReboot = kernelVersion, needRebootFlag
	return ko, nil
}

func KexecCmdline(km *KernelMessage) string {
	return fmt.Sprintf("-l --reuse-cmdline --initrd /boot/initrd.img-%[1]s /boot/vmlinuz-%[1]s",
		km.KernelVersion)
}

//Kexec load kernel and execute function
func IsNeedKernReboot(kmObject *KernelMessage, flags FlagSet) bool {
	if !kmObject.NeedReboot && flags.Job == UPGRADE {
		fmt.Println("kernel upgrade reboot not needed")
		return false
	}
	return true
}
