import re
import time
import subprocess32 as subprocess
import json
import errors
import os
import semver
from library.python.svn_version import svn_revision
from juggler.bundles import Event, Status

HW_WATCHER_PATH = "/usr/sbin/hw_watcher"

WALLE_STATE_DIR = "/home/monitor/walle"
MEMORY_REPAIR_FILE_PATH = os.path.join(WALLE_STATE_DIR, "memory_repair_time")


class HWWatcherModuleDisabled(Exception):
    def __init__(self, module):
        super(HWWatcherModuleDisabled, self).__init__()
        self.module = module

    def __str__(self):
        return "hwwatcher module '{}' is disabled".format(self.module)


class HWWatcherNonJsonOutput(Exception):
    def __init__(self, hww_output):
        super(HWWatcherNonJsonOutput, self).__init__("hwwatcher returned non-json output: {}".format(hww_output))
        self.hww_output = hww_output


class HWWatcherError(Exception):
    def __init__(self, stderr):
        super(HWWatcherError, self).__init__(stderr)


def read_content(filename):
    with open(filename, "rt") as src:
        return src.read()


def get_command_output(args, timeout=30):
    process = subprocess.Popen(
        args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True
    )

    try:
        stdout, stderr = process.communicate(timeout=timeout)
    except subprocess.TimeoutExpired:
        process.kill()
        stdout, stderr = process.communicate()
        error = stderr.strip() or stdout.strip()
        raise subprocess.TimeoutExpired(process.args, timeout, output=error.decode("utf-8")[:800])

    return_code = process.poll()
    if return_code or stderr:
        error = stderr.strip() or stdout.strip()
        raise subprocess.CalledProcessError(return_code, process.args, output=error.decode("utf-8")[:800])

    return stdout


def get_hw_watcher_status(check_type):
    try:
        stdout = get_command_output([HW_WATCHER_PATH, check_type, "extended_status"])
    except subprocess.CalledProcessError as e:
        raise HWWatcherError(e.output)
    else:
        try:
            return json.loads(stdout)
        # corner case for disabled module message: hww doesn't raise, but its output is not a valid json
        except ValueError as e:
            handle_json_value_error(e, stdout)


def handle_json_value_error(exc, stdout):
    if exc.message == "No JSON object could be decoded":
        m = re.match(r"OK; Module (?P<disabled_module>.+?) disabled", stdout)
        if m:
            raise HWWatcherModuleDisabled(m.group("disabled_module"))
        else:
            raise HWWatcherNonJsonOutput(truncate_description_string(stdout))
    raise exc


def oldstyle_main(check_name, event):
    print("PASSIVE-CHECK:{};{};{}".format(check_name, event.status, event.description))


def get_bundle_version():
    return svn_revision()


def timestamp():
    return int(time.time())


def make_event(status, msg):
    return Event(status, json.dumps(msg))


def truncate_description_string(desc, maxlen=700):
    shorten_str = ' ... '

    if len(desc) > maxlen:
        idx = int(maxlen/2) - int(len(shorten_str)/2)
        return desc[:idx] + shorten_str + desc[(len(desc) - idx + 1):]

    else:
        return desc


def make_warning_event(exception):
    return make_event(Status.WARN, {"result": {"reason": ["Can't get status from hw-watcher: {}".format(exception)],
                                               "timestamp": timestamp()}})


def get_cpu_model_name(open_func=open):
    with open_func('/proc/cpuinfo', 'r') as f:
        for line in f:
            if line.startswith('model name'):
                return line.split(':')[1].strip()

    raise errors.CheckError('cannot determine CPU model. There is no "model name" in /proc/cpuinfo')


def get_kernel_version():
    return semver.parse(get_command_output(["/bin/uname", "-r"]))
