import itertools


class CpuNumbersRange(object):
    __slots__ = ("start", "end")

    def __init__(self, start, end=None):
        """Represent a range of numbers with both end included."""

        if end is None:
            end = start
        self.start = start
        self.end = end

    @classmethod
    def parse(cls, range_string):
        return cls(*list(map(int, range_string.split("-"))))

    def intersection(self, other):
        if other.start < self.start:
            start = self.start
        else:
            start = other.start

        if other.end > self.end:
            end = self.end
        else:
            end = other.end
        if start > end:
            return None
        else:
            return CpuNumbersRange(start, end)

    def __iter__(self):
        return iter(xrange(self.start, self.end + 1))

    def _as_tuple(self):
        return self.start, self.end

    def as_string(self):
        if self.start == self.end:
            return str(self.start)
        else:
            return "{}-{}".format(self.start, self.end)

    def __cmp__(self, other):
        return cmp(self._as_tuple(), other._as_tuple())

    def __lt__(self, other):
        # for sorting
        return cmp(self, other) < 0

    def __gt__(self, other):
        # for sorting
        return cmp(self, other) > 0

    def __repr__(self):
        return "{}({})".format(type(self), self.as_string())


class RangeList(object):
    __slots__ = ("ranges_list", )

    def __init__(self, ranges):
        self.ranges_list = sorted(ranges)

    @classmethod
    def parse(cls, cores_ranges_string):
        return RangeList(CpuNumbersRange.parse(r) for r in cores_ranges_string.split(","))

    def as_string(self):
        self.compress()
        return ",".join(r.as_string() for r in self.ranges_list)

    def intersection(self, other):
        iter_self = iter(self)
        iter_other = iter(other)

        collector = []
        stacked = None

        start = None
        end = None
        for r1 in iter_self:
            if stacked is not None:
                if stacked < r1:
                    stacked = None
                    continue
                if stacked == r1:
                    start = r1
                    end = r1
                    stacked = None
                    continue
                if stacked > r1:
                    continue

            for r2 in iter_other:
                if r2 < r1:
                    continue
                if r2 == r1:
                    if start is None:
                        start = r1
                        end = r1
                    elif r2 > end + 1:
                        collector.append(CpuNumbersRange(start, end))
                        start = r1
                        end = r1
                    else:
                        end = r2
                    break
                if r2 > r1:
                    stacked = r2
                    if start is not None:
                        collector.append(CpuNumbersRange(start, end))
                        start = None
                        end = None
                    break

        if start is not None:
            collector.append(CpuNumbersRange(start, end))

        if collector:
            return type(self)(collector)
        else:
            return None

    def __iter__(self):
        return itertools.chain.from_iterable(iter(r) for r in self.ranges_list)

    def compress(self):
        start = None
        end = None
        new_ranges_list = []
        for i in self:
            if start is None:
                start = i
                end = i
                continue
            if i <= end + 1:
                end = max(i, end)
                continue
            else:
                new_ranges_list.append(CpuNumbersRange(start, end))
                start = i
                end = i

        new_ranges_list.append(CpuNumbersRange(start, end))
        self.ranges_list = new_ranges_list

    def __cmp__(self, other):
        return cmp(self.ranges_list, other.ranges_list)

    def __repr__(self):
        return "{}({})".format(type(self), self.as_string())
