#!/usr/bin/env python
from __future__ import absolute_import, print_function

import itertools
import json
import logging
import os
import re
import sys
import time

import requests
import datetime
import click

from debian.changelog import Changelog


log = logging.getLogger(__name__)

CRD_SANDBOX_OAUTH_TOKEN = "sandbox_oauth_token"
CRD_SANDBOX_USERNAME = "sandbox_username"
CRD_TOKENS = {CRD_SANDBOX_OAUTH_TOKEN, CRD_SANDBOX_USERNAME}
CRD_FILENAME_CWD = ".release.rc"
CRD_FILENAME_HOME = os.path.expanduser("~/.release.rc")
SANDBOX_TASK_OWNER = 'WALLE'

SANDBOX_STATUSES_SUCCESS = {"SUCCESS", "RELEASED"}
SANDBOX_STATUSES_FAILED = {"ERROR", "FAILURE", "EXCEPTION", "TIMEOUT", "STOPPED", "NO_RES", "EXPIRED", "DELETED"}


def read_credentials_from_config_file(rc_filename, verbose=False):
    credentials = {}
    if not os.path.isfile(rc_filename):
        return credentials

    try:
        with open(rc_filename) as f:
            for line in f.read().splitlines():
                if not line:
                    continue

                k, v = map(str.strip, line.split(':'))
                if k in CRD_TOKENS:
                    credentials[k] = v
    except IOError as e:
        if verbose:
            print("Failed to read credentials from %s: %s", rc_filename, e)
    except (KeyError, ValueError):
        if verbose:
            print("Incorrect RC file format [%s]", rc_filename)

    return credentials


def get_credentials_from_env():
    env_names = {"SANDBOX_OAUTH_TOKEN": CRD_SANDBOX_OAUTH_TOKEN, "SANDBOX_USERNAME": CRD_SANDBOX_USERNAME}
    credentials = {}

    for var in env_names:
        if var in os.environ:
            credentials[env_names[var]] = os.environ[var]

    return credentials


def get_credentials_from_args(sandbox_oauth_token=None, sandbox_username=None):
    credentials = {}

    if sandbox_oauth_token:
        credentials[CRD_SANDBOX_OAUTH_TOKEN] = sandbox_oauth_token
    if sandbox_username:
        credentials[CRD_SANDBOX_USERNAME] = sandbox_username

    return credentials


def add_missing(original, other):
    for key in other.viewkeys() - original.viewkeys():
        original[key] = other[key]


def get_credentials(sandbox_oauth_token=None, sandbox_username=None, verbose=False):
    credentials = get_credentials_from_args(sandbox_oauth_token, sandbox_username)

    if credentials.viewkeys() < CRD_TOKENS:
        add_missing(credentials, get_credentials_from_env())

    if credentials.viewkeys() < CRD_TOKENS:
        add_missing(credentials, read_credentials_from_config_file(CRD_FILENAME_CWD, verbose))

    if credentials.viewkeys() < CRD_TOKENS:
        add_missing(credentials, read_credentials_from_config_file(CRD_FILENAME_HOME, verbose))

    if credentials.viewkeys() < CRD_TOKENS:
        print("Please, provide sandbox oauth token and sandbox username", file=sys.stderr)
        exit(2)

    return credentials


def sandbox(method, path, oauth_token, **kwargs):
    url = "https://sandbox.yandex-team.ru{}".format(path)
    resp = requests.request(method, url, data=json.dumps(kwargs), headers={
        "Authorization": "OAuth {}".format(oauth_token),
        "Content-Type": "application/json"
    })
    return resp.json() if resp.content else None


def start_building_juggler_bundle(credentials):
    task_id = sandbox(
        "POST", "/api/v1.0/task",
        oauth_token=credentials[CRD_SANDBOX_OAUTH_TOKEN],
        owner=SANDBOX_TASK_OWNER,
        type="BUILD_JUGGLER_CHECKS_BUNDLE",
        description="Build Wall-E checks bundle",
        priority={
            "class": "USER",
            "subclass": "NORMAL"
        },
        context={
            "package_path": "infra/wall-e/checks/bundle-juggler/bundle.json",
            "bundle_name": "wall-e-checks-bundle",
            "upload_to_mds": True
        }
    )["id"]

    # Looks like batch task start is the only way to start a task.
    sandbox(
        "PUT", "/api/v1.0/batch/tasks/start",
        oauth_token=credentials[CRD_SANDBOX_OAUTH_TOKEN],
        comment="Build Wall-E checks bundle",
        id=[task_id]
    )
    return task_id


def start_building_deb_bundle(credentials):
    task_id = sandbox(
        "POST", "/api/v1.0/task",
        oauth_token=credentials[CRD_SANDBOX_OAUTH_TOKEN],
        owner=SANDBOX_TASK_OWNER,
        type="YA_PACKAGE",
        description="Build Wall-E checks bundle deb",
        priority={
            "class": "USER",
            "subclass": "NORMAL"
        },
        context={
            "packages": "infra/wall-e/checks/bundle-deb/bundle.json",
            "package_type": "debian",
            "publish_to": "search",
            "key_user": "robot-walle",
        }
    )["id"]

    # Looks like batch task start is the only way to start a task.
    sandbox(
        "PUT", "/api/v1.0/batch/tasks/start",
        oauth_token=credentials[CRD_SANDBOX_OAUTH_TOKEN],
        comment="Build Wall-E checks bundle",
        id=[task_id]
    )
    return task_id


def get_sandbox_task_status(task_id, credentials):
    return sandbox(
        "GET", "/api/v1.0/task/{}".format(task_id),
        oauth_token=credentials[CRD_SANDBOX_OAUTH_TOKEN],
    )["status"]


def wait_for_sandbox_task(task_id, credentials):
    with Spinner() as spinner:
        while True:
            status = get_sandbox_task_status(task_id, credentials)
            spinner.spin("{}: {}".format(task_id, status))

            if status in SANDBOX_STATUSES_SUCCESS | SANDBOX_STATUSES_FAILED:
                return status

            time.sleep(10)


class Spinner(object):
    def __init__(self):
        self.char = itertools.cycle("|/-\\")
        self.prefix = ""

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        # clear last spinner character. Add new line so that next written line went where it is supposed to go.
        sys.stdout.write("\b \n")

    def spin(self, prefix=None):
        if prefix is not None:
            self.set_new_prefix(prefix)

        sys.stdout.write("\b" + next(self.char))
        sys.stdout.flush()

    def set_new_prefix(self, prefix):
        new_prefix = prefix + "  "  # one space between status and spinner and one space for spinner itself
        if new_prefix != self.prefix:
            sys.stdout.write("\r" + (" " * len(self.prefix)))
            self.prefix = new_prefix
            sys.stdout.write("\r" + self.prefix)


def task_url(task_id):
    return "https://sandbox.yandex-team.ru/task/{}/view".format(task_id)


def report_task(task_id, status, verbose=False):
    if status in {"SUCCESS", "RELEASED"}:
        if verbose:
            print("Done.")
            exit(0)
    else:
        print(
            "Failed to complete task [{}]. Visit task page for details: {}".format(status, task_url(task_id)),
            file=sys.stderr
        )
        exit(16)


@click.group()
def release():
    pass


@release.command("bump-changelog", help="Bump debian changelog")
@click.option("-p", "--changelog-path", default="bundle-deb/debian/changelog")
@click.option("-v", "--version", required=False, help="Version dotted string")
@click.option("-a", "--author", required=False, help="Author name in form 'Name Surname <name.surname@company.org>'")
@click.option("-m", "--message", default=["New upstream release"], multiple=True)
def bump_changelog(changelog_path, message, version=None, author=None):
    with open(changelog_path) as changelog_file:
        changelog = Changelog(changelog_file)

    if author:
        if not re.match(r'^([\w\d_.-]+\s)+<[\w\d_.-]+@[\w\d.-]+>$', author):
            print("'Author' name string must include name, surname and e-mail in a form like")
            print("Name Surname <name.surname@company.org>")
            print("Look for examples in the changelog file {}".format(changelog_path))
            exit(1)
    else:
        author = changelog.author

    if version is None:
        last_version = sorted(changelog.versions)[-1]
        version_parts = last_version.full_version.split(".")
        minor_version = int(version_parts[-1]) + 1
        version = ".".join(version_parts[:-1] + [str(minor_version)])

    changelog.new_block(
        package=changelog.package, version=version, distributions=changelog.distributions,
        urgency=changelog.urgency, author=author,
        date=datetime.datetime.utcnow().strftime("%a, %d %b %Y %H:%M:%S +0000"),
        changes=[""] + ["  * {}".format(msg) for msg in message] + [""]
    )

    changelog_data = str(changelog)

    with open(changelog_path, "w") as changelog_file:
        changelog_file.write(changelog_data)

    print("Change log bump was successful. Now go revisit change message in {}".format(changelog_path))
    print("And don't forget to commit it!")


@release.command("release-bundle", help="Build Wall-E checks juggler-bundle")
@click.option("-t", "--sandbox-oauth-token", required=False)
@click.option("-u", "--sandbox-username", required=False, help="Sandbox task owner")
@click.option("-w", "--wait", default=True, help="Wait for task to complete")
@click.option("-v", "--verbose", default=False, help="Don't keep silent, print something")
def release_bundle(sandbox_oauth_token=None, sandbox_username=None, wait=True, verbose=False):
    credentials = get_credentials(sandbox_oauth_token, sandbox_username, verbose)

    task_id = start_building_juggler_bundle(credentials)

    if verbose or not wait:
        print("Task URL is {}".format(task_url(task_id)))

    if wait:
        status = wait_for_sandbox_task(task_id, credentials)
        report_task(task_id, status, verbose)


@release.command("release-deb", help="Build Wall-E checks deb package")
@click.option("-t", "--sandbox-oauth-token", required=False)
@click.option("-u", "--sandbox-username", required=False, help="Sandbox task owner")
@click.option("-w", "--wait", default=True, help="Wait for task to complete")
@click.option("-v", "--verbose", default=False, help="Don't keep silent, print something")
def release_deb(sandbox_oauth_token=None, sandbox_username=None, wait=True, verbose=False):
    credentials = get_credentials(sandbox_oauth_token, sandbox_username, verbose)

    task_id = start_building_deb_bundle(credentials)

    if verbose or not wait:
        print("Task URL is {}".format(task_url(task_id)))

    if wait:
        status = wait_for_sandbox_task(task_id, credentials)
        report_task(task_id, status, verbose)


if __name__ == "__main__":
    release()
