#!/usr/bin/env python

# Provides: walle_cpu

from __future__ import absolute_import

import json
from functools import wraps

from juggler.bundles import as_check, Status, Event
from .common import oldstyle_main, read_content
from .number_ranges import RangeList

CHECK_NAME = "walle_cpu"


class UnsupportedPlatform(Exception):
    pass


def on_unsupported_platform(msg_format, silent=False):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if silent:
                    return msg_format.format(e)
                else:
                    raise UnsupportedPlatform(msg_format.format(e))
        return wrapper

    return decorator


def run_check():
    online_cores = read_online_cores()
    present_cores = read_present_cores()

    if online_cores != present_cores:
        present_ranges = RangeList.parse(present_cores)
        offline_ranges = RangeList.parse(read_offline_cores())
        present_offline_cores = present_ranges.intersection(offline_ranges)

        if present_offline_cores:
            return Status.CRIT, "offline cores: " + present_offline_cores.as_string()
        else:
            return Status.WARN, "online cores ({}) != present cores ({}), but there are no present offline cores ({})".format(
                online_cores, present_cores, offline_ranges.as_string()
            )
    else:
        return Status.OK, "Ok"


@on_unsupported_platform("can not read offline cores number: {}")
def read_offline_cores():
    return read_content("/sys/devices/system/cpu/offline").strip()


@on_unsupported_platform("can not read present cores number: {}")
def read_present_cores():
    return read_content("/sys/devices/system/cpu/present").strip()


@on_unsupported_platform("can not read online cores number: {}")
def read_online_cores():
    return read_content("/sys/devices/system/cpu/online").strip()


@on_unsupported_platform("can not read board name: {}", silent=True)
def read_board_name():
    return read_content("/sys/class/dmi/id/board_name").strip()


@on_unsupported_platform("can not read cpuinfo: {}", silent=True)
def read_processor_vendor():
    for line in read_content("/proc/cpuinfo").splitlines():
        if line.startswith('vendor_id'):
            return line.split(':')[1].strip()

    return "Can not find out cpu vendor, no vendor_id string in /proc/cpuinfo"


def make_event(status, msg, board, cpu_vendor):
    return Event(status, json.dumps({"result": {
        "reason": [msg],
        "board": board,
        "cpu_vendor": cpu_vendor,
    }}))


@as_check(name=CHECK_NAME)
def juggler_check():
    board = read_board_name()
    cpu_vendor = read_processor_vendor()

    try:
        status, msg = run_check()
        return make_event(status, msg, board, cpu_vendor)
    except Exception as e:
        return make_event(Status.WARN, "Failed to get data for walle_cpu check: {}".format(e), board, cpu_vendor)


if __name__ == '__main__':
    oldstyle_main(CHECK_NAME, juggler_check())
