#!/usr/bin/env python2

# Provides: walle_cpu_capping

# HWPROBLEMS-7 Check CPU frequency capping

from __future__ import print_function, absolute_import
import errno
import re
from multiprocessing import cpu_count
from os.path import join

from .common import oldstyle_main, read_content, make_event, timestamp, get_cpu_model_name, get_kernel_version
from juggler.bundles import as_check, Status


class UnsupportedKernel(Exception):
    def __init__(self, reason=None):
        msg = "This kernel doesn't support capping detection"
        if reason is not None:
            msg += ": {}".format(reason)

        super(UnsupportedKernel, self).__init__(msg)


CHECK_NAME = "walle_cpu_capping"
ROOT_SYS_CPU = "/sys/devices/system/cpu"


def get_cpu_max_freq_and_limit(cpuid):
    bios_limit_filepath = join(ROOT_SYS_CPU, "cpu{}".format(cpuid), "cpufreq", "bios_limit")
    cpufreq_max_filepath = join(ROOT_SYS_CPU, "cpu{}".format(cpuid), "cpufreq", "cpuinfo_max_freq")
    try:
        return int(read_content(cpufreq_max_filepath)), int(read_content(bios_limit_filepath))
    except IOError as e:
        if e.errno == errno.ENOENT:
            raise UnsupportedKernel("File not found: {}".format(e.filename))
        raise


def cpu_supported(open_func=open):
    unsupported_re = re.compile(r"^.*Xeon.*E5645.*$")

    if unsupported_re.match(get_cpu_model_name(open_func)):
        return False

    return True


def assert_kernel_is_supported():
    version = get_kernel_version()
    if version["major"] < 4:
        raise UnsupportedKernel("Kernel versions < 4.9 are not supported")


def capping_detect(open_func=open):
    if cpu_supported(open_func):
        capped_cpus = []
        min_bios_limit = 2 ** 64 - 1
        for cpuid in range(cpu_count()):
            max_freq, bios_limit = get_cpu_max_freq_and_limit(cpuid)
            if bios_limit < max_freq:
                capped_cpus.append(cpuid)
                if min_bios_limit > bios_limit:
                    min_bios_limit = bios_limit

        if capped_cpus:
            description = "Capping detected on {} CPUs, min_freq: {} GHz".format(
                len(capped_cpus), min_bios_limit / 1000000
            )
            return Status.CRIT, description

    return Status.OK, "Ok"


@as_check(name=CHECK_NAME)
def juggler_check(open_func=open):
    try:
        assert_kernel_is_supported()
        check_exit_status, description = capping_detect(open_func)
    except UnsupportedKernel as e:
        check_exit_status, description = Status.OK, e.message
    except Exception as e:
        check_exit_status = Status.WARN
        description = "Can't get check data: {}".format(e)

    metadata = {
        "timestamp": timestamp(),
        "reason": description
    }

    return make_event(check_exit_status, metadata)


if __name__ == '__main__':
    oldstyle_main(CHECK_NAME, juggler_check())
