#!/usr/bin/env python2

# Provides: walle_reboots

from __future__ import print_function, absolute_import

import errno
import glob
import json
import mmap
import os
import struct
import time
from collections import namedtuple, OrderedDict

from .common import oldstyle_main, MEMORY_REPAIR_FILE_PATH
from juggler.bundles import as_check, Status, Event

CHECK_NAME = "walle_reboots"

DURATION_HOUR = 60 * 60
DURATION_DAY = 60 * 60 * 24
DURATION_WEEK = 7 * DURATION_DAY

RESULT = 'result'
COUNT = 'count'
REASON = 'reason'

REBOOT = 'REBOOT'
SHUTDOWN = 'SHUTDOWN'
UNKNOWN = 'UNKNOWN'
REBOOT_OR_SHUTDOWN = {REBOOT, SHUTDOWN}

LimitConfig = namedtuple('LimitConfig', 'limit, period, description')

limits = [
    LimitConfig(limit=3, period=2*DURATION_DAY, description="per two days"),
    LimitConfig(limit=5, period=DURATION_WEEK, description="per week")
]

XTMP_Event = namedtuple('XTMP_Event', 'type, timestamp')


# / *Values
# for ut_type field, below * /
#
#     # define EMPTY         0 /* Record does not contain valid info
#     (formerly known as UT_UNKNOWN on Linux) * /
#     # define RUN_LVL       1 /* Change in system run-level (see
#     init(8)) * /
#     # define BOOT_TIME     2 /* Time of system boot (in ut_tv) */
#     # define NEW_TIME      3 /* Time after system clock change
#     (in ut_tv) * /
#     # define OLD_TIME      4 /* Time before system clock change
#     (in ut_tv) * /
#     # define INIT_PROCESS  5 /* Process spawned by init(8) */
#     # define LOGIN_PROCESS 6 /* Session leader process for user login */
#     # define USER_PROCESS  7 /* Normal process */
#     # define DEAD_PROCESS  8 /* Terminated process */
#     # define ACCOUNTING    9 /* Not implemented */
#
#     # define UT_LINESIZE      32
#     # define UT_NAMESIZE      32
#     # define UT_HOSTSIZE     256
#
# struct exit_status {              /* Type for ut_exit, below */
#    short int e_termination;      /* Process termination status */
#    short int e_exit;             /* Process exit status */
# };
#
# struct utmp {
#    short   ut_type;              /* Type of record */
#    pid_t   ut_pid;               /* PID of login process */
#    char    ut_line[UT_LINESIZE]; /* Device name of tty - "/dev/" */
#    char    ut_id[4];             /* Terminal name suffix,
#                                     or inittab(5) ID */
#    char    ut_user[UT_NAMESIZE]; /* Username */
#    char    ut_host[UT_HOSTSIZE]; /* Hostname for remote login, or
#                                     kernel version for run-level
#                                     messages */
#    struct  exit_status ut_exit;  /* Exit status of a process
#                                     marked as DEAD_PROCESS; not
#                                     used by Linux init(8) */
#    /* The ut_session and ut_tv fields must be the same size when
#       compiled 32- and 64-bit.  This allows data files and shared
#       memory to be shared between 32- and 64-bit applications. */
# #if __WORDSIZE == 64 && defined __WORDSIZE_COMPAT32
#    int32_t ut_session;           /* Session ID (getsid(2)),
#                                     used for windowing */
#    struct {
#        int32_t tv_sec;           /* Seconds */
#        int32_t tv_usec;          /* Microseconds */
#    } ut_tv;                      /* Time entry was made */
# #else
#     long   ut_session;           /* Session ID */
#     struct timeval ut_tv;        /* Time entry was made */
# #endif
#
#    int32_t ut_addr_v6[4];        /* Internet address of remote
#                                     host; IPv4 address uses
#                                     just ut_addr_v6[0] */
#    char __unused[20];            /* Reserved for future use */
# };


XTMP_STRUCT = 'hi32s4s32s256shhiii4i20x'
XTMP_STRUCT_SIZE = struct.calcsize(XTMP_STRUCT)


def get_event_type(record_type, login):
    if record_type == 2:  # record type BOOT_TIME
        return REBOOT

    if login == 'shutdown':  # event: user shutdown
        return SHUTDOWN

    return UNKNOWN


def transform_xtmp_struct_to_namedtuple(xtmp_struct):
    record_type = xtmp_struct[0]

    login = xtmp_struct[4]
    login = login[0:login.find('\0')]

    timestamp = xtmp_struct[9]

    return XTMP_Event(type=get_event_type(record_type, login), timestamp=timestamp)


def read_xtmp(f_name):
    try:
        with open(f_name, 'rb') as f:
            # File is empty or broken - nothing to do
            size = os.fstat(f.fileno()).st_size
            if size < XTMP_STRUCT_SIZE:
                return
            # Read struct after struct backwards from the file
            mapping = mmap.mmap(f.fileno(), 0, access=mmap.PROT_READ)
            offset = mapping.size() - XTMP_STRUCT_SIZE
            while offset >= 0:
                data = mapping[offset:offset+XTMP_STRUCT_SIZE]
                if len(data) < XTMP_STRUCT_SIZE:
                    break
                unpacked_struct = struct.unpack(XTMP_STRUCT, data)
                event = transform_xtmp_struct_to_namedtuple(unpacked_struct)
                yield event
                offset -= XTMP_STRUCT_SIZE
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            return
        raise


def parse_last_logs():
    list_of_files = ['/var/log/wtmp'] + glob.glob("/var/log/wtmp.?")
    for f_name in list_of_files:
        for event in read_xtmp(f_name):
            yield event


def _get_memory_repairing_time():
    try:
        with open(MEMORY_REPAIR_FILE_PATH) as f:
            return int(f.read())
    except Exception:
        return 0


def fetch_records_for_last_week():
    period = max(time.time() - DURATION_WEEK, _get_memory_repairing_time())
    records = []
    for record in parse_last_logs():
        if record.timestamp < period:  # tv_sec
            # Don't look too far in time
            break
        if record.type in REBOOT_OR_SHUTDOWN:
            records.append(record)
    return records


def count_reboots(records, cutoff_time):
    c = 0
    for record in records:
        if record.timestamp < cutoff_time:
            break
        if record.type == REBOOT:
            c += 1
        elif record.type == SHUTDOWN:
            c -= 1
        else:
            pass

    return c


def make_check_result(reboots, reboot_count):
    reboots_str = ", ".join("{} {}".format(v, k) for k, v in reboots.items())

    return {RESULT: {
        COUNT: reboot_count or max(reboots.values()),
        REASON: "Reboot counts: {}.".format(reboots_str),
    }}


def run_walle_reboots_check():
    status = Status.OK
    reboot_count = 0
    records = fetch_records_for_last_week()
    data = OrderedDict()

    for limit in limits:
        cutoff_time = time.time() - limit.period
        c = count_reboots(records, cutoff_time)
        data[limit.description] = max(c, 0)  # prevent behaviour, when check shows count as -1

        if c >= limit.limit:
            status = Status.CRIT
            reboot_count = max(c, reboot_count)

    result = make_check_result(data, reboot_count)
    return status, result


@as_check(name=CHECK_NAME)
def juggler_check():
    status, result = run_walle_reboots_check()
    return Event(status, json.dumps(result))


if __name__ == '__main__':
    oldstyle_main(CHECK_NAME, juggler_check())
