"""Access token management."""

import errno
import os

from walle_cli.common import Error, get_config_path, register_subparsers, register_parser


def get_token():
    token_path = _get_token_path()

    try:
        with open(token_path) as token_file:
            return token_file.read().strip()
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            return

        raise Error("Failed to read access token file '{}': {}.", token_path, e.strerror)


def init(subparsers):
    subparsers = register_subparsers(subparsers, "access-token", "Access token management", short="t")
    register_parser(subparsers, "show", _on_show, "Print saved access token")
    register_parser(subparsers, "get", _on_get, "Obtain and print an access token, but not save it to the client")
    register_parser(subparsers, "obtain", _on_obtain, "Obtain an access token and save it to the client")
    register_parser(subparsers, "remove", _on_remove, "Remove an obtained access token from the client", short="d")


def _on_show(client, args):
    token = get_token()
    if token is None:
        raise Error("Access token hasn't been obtained yet.")

    print("Obtained access token:", token)


def _on_get(client, args):
    token = _obtain(client)
    print("Obtained access token:", token)


def _on_obtain(client, args):
    token = _obtain(client)
    _save(token)
    print("Access token has been successfully obtained.")


def _on_remove(client, args):
    token_path = _get_token_path()

    try:
        os.unlink(token_path)
    except EnvironmentError as e:
        if e.errno != errno.ENOENT:
            raise Error("Failed to remove access token file '{}': {}.", token_path, e.strerror)


def _obtain(client):
    print("Please go to " + client.get_authorization_url())

    authorization_code = input("Enter obtained confirmation code: ").strip()
    return client.obtain_access_token(authorization_code)


def _save(token):
    token_path = _get_token_path(ensure_parent_exists=True)

    def opener(path, flags):
        return os.open(path, flags, mode=0o600)

    try:
        with open(token_path, "w", opener=opener) as token_file:
            token_file.write(token)
    except EnvironmentError as e:
        raise Error("Failed to save access token to '{}': {}.", token_path, e.strerror)


def _get_token_path(ensure_parent_exists=False):
    config_path = get_config_path(ensure_exists=ensure_parent_exists)
    return os.path.join(config_path, "access_token")
