"""Tools for argparse"""
from argparse import Action, ArgumentError

from walle_cli.common import parse_limit


class ActionStoreCredit(Action):
    def __call__(self, parser, namespace, values, option_string=None):
        credits = _set_default(namespace, self.dest, {})

        check_name, credit = values
        limit_name, credit = self._convert_values(check_name, credit)

        if limit_name in credits:
            raise ArgumentError(self, "multiple credit values for check {}".format(check_name))

        credits[limit_name] = credit

    def _convert_values(self, check_name, credit):
        limit_name = _limit_name(check_name)

        try:
            credit = int(credit)
        except (TypeError, ValueError):
            raise ArgumentError(self, "invalid credit value for check {}: {!r}".format(check_name, credit))

        return limit_name, credit


class ActionAppendLimit(Action):
    def __call__(self, parser, namespace, values, option_string=None):
        limit_values = _set_default(namespace, self.dest, {})

        limit_name, limit = self._convert_values(values)
        limit_values.setdefault(limit_name, []).append(limit)

    def _convert_values(self, values):
        check_name, limit = values
        limit_name = _limit_name(check_name)

        try:
            limit = parse_limit(limit)
        except (TypeError, ValueError):
            raise ArgumentError(self, "invalid limit specification for check {}: {!r}.".format(check_name, limit))

        return limit_name, limit


class StoreKeyValuePair(Action):
    def __call__(self, parser, namespace, values, option_string=None):
        if isinstance(values, str):
            values = [values]

        items = _set_default(namespace, self.dest, {})
        try:
            items.update(v.split("=") for v in values)
        except ValueError:
            parser.error("Arguments for KeyValuePair must be represented as <key>=<value>."
                         "If the argument allows you to enter several pairs of values, then separate them with a space")


def _set_default(namespace, name, value):
    if getattr(namespace, name, None) is None:
        setattr(namespace, name, value)

    return getattr(namespace, name)


def _limit_name(check_name):
    return "max_{}_failures".format(check_name)
