"""Audit log management."""

import sys
from collections import deque

try:
    import json
except ImportError:
    import simplejson as json

from walle_cli.common import register_subparsers, register_parser, get_supported_time_formats, parse_time, format_time


def init(subparsers):
    subparsers = register_subparsers(subparsers, "audit-log", "Audit log management")

    parser = register_parser(subparsers, "get", _on_get, "Get the log")

    parser.add_argument("-a", "--issuer", help="filter by issuers separated by commas")
    parser.add_argument("-t", "--event-type", help="filter by event types separated by commas")
    parser.add_argument("-p", "--project", help="filter by project IDs separated by commas")
    parser.add_argument("-P", "--automation-plot", help="filter by automation plot ID's separated by commas")
    parser.add_argument("-sc", "--scenario", help="filter by scenario ID's separated by commas")
    parser.add_argument("-i", "--host-inv", help="filter by host inventory numbers separated by commas")
    parser.add_argument("-n", "--host-name", help="filter by host names separated by commas")
    parser.add_argument("-u", "--host-uuid", help="filter by host UUIDs separated by commas")
    parser.add_argument("-S", "--status", help="filter by task status separated by commas")

    supported_formats = get_supported_time_formats()
    parser.add_argument("-s", "--start-time", help="time lower bound in the following format: " + supported_formats)
    parser.add_argument("-e", "--end-time", help="time upper bound in the following format: " + supported_formats)

    parser.add_argument("-r", "--show-reason", action="store_true", help="show event reason")
    parser.add_argument("-d", "--show-payload", action="store_true", help="show event payload")


def _on_get(client, args):
    start_time = parse_time(args.start_time, future_period=False)
    end_time = parse_time(args.end_time, future_period=False)

    fields = ["time", "issuer", "type",
              "automation_plot", "project", "scenario_id", "host_inv", "host_name", "host_uuid",
              "status", "status_time", "error"]
    if args.show_reason:
        fields.append("reason")
    if args.show_payload:
        fields.append("payload")

    if start_time is None and end_time is None:
        reverse = True
        limit = 100
    else:
        reverse = False
        limit = None

    entries = client.iter_audit_log(
        issuer=args.issuer, project=args.project, automation_plot=args.automation_plot,
        host_inv=args.host_inv, host_name=args.host_name, host_uuid=args.host_uuid,
        scenario_id=args.scenario, event_type=args.event_type,
        start_time=start_time, end_time=end_time, limit=limit, status=args.status,
        fields=fields, reverse=reverse
    )

    if reverse:
        entries = list(entries)
        entries.reverse()

    entry_page = deque()

    for entry in entries:
        entry_page.append(entry)

        if len(entry_page) >= client.MAX_PAGE_SIZE:
            _display_page(entry_page, args)
            entry_page.clear()

    if entry_page:
        _display_page(entry_page, args)

    if limit is not None and not args.batch:
        print("> Listed last {limit} entries. "
              "Use --start-time and --end-time to list all entries for the specified period.".format(limit=limit), file=sys.stderr)


def _display_page(entries, args):
    issuer_len = max(len(entry["issuer"]) for entry in entries)
    type_len = max(len(entry["type"]) for entry in entries)
    project_len = max([len(entry["project"]) for entry in entries if "project" in entry] or [0])
    plot_id_len = max([len(entry["automation_plot"]) for entry in entries if "automation_plot" in entry] or [0])

    for entry in entries:
        log_line = ("{time} [{issuer:>"+str(issuer_len)+"}] [{type:>"+str(type_len)+"}]").format(
            time=format_time(entry["time"]), issuer=entry["issuer"], type=entry["type"])

        if args.automation_plot is None or "," in args.automation_plot:
            if "automation_plot" in entry:
                log_line += (" [{automation_plot:>"+str(plot_id_len)+"}]").format(automation_plot=entry["automation_plot"])
            elif plot_id_len:
                log_line += (" [{automation_plot:"+str(plot_id_len)+"}]").format(automation_plot="")

        if args.project is None or "," in args.project:
            if "project" in entry:
                log_line += (" [{project:>"+str(project_len)+"}]").format(project=entry["project"])
            elif project_len:
                log_line += (" [{project:"+str(project_len)+"}]").format(project="")

        inv_part = "#{}".format(entry["host_inv"]) if "host_inv" in entry else None

        host_id_section = " [{}]".format(
            "|".join(part for part in (inv_part, entry.get("host_name"), entry.get("host_uuid")) if part is not None)
        )
        log_line += host_id_section

        # if "host_inv" in entry:
        #     log_line += " [#" + str(entry["host_inv"])
        #     if "host_name" in entry:
        #         log_line += "|" + entry["host_name"]
        #     log_line += "]"

        log_line += " -> {status_time} [{status}]".format(
            status=entry["status"], status_time=format_time(entry["status_time"]))

        if args.show_reason and "reason" in entry:
            log_line += " ({reason})".format(reason=entry["reason"])

        if "error" in entry:
            log_line += ": " + entry["error"]

        print(log_line)
        if args.show_payload and entry.get("payload"):
            print(json.dumps(entry["payload"], indent=2))
