from walle_cli.common import Error, register_subparsers, register_parser, parse_arg_list, render_result, str2bool


AUTOMATION_PLOT_FIELDS = ["id", "owners", "name", "checks"]
AUTOMATION_PLOT_LIST_FIELDS = ["id", "owners", "name"]


def init(subparsers):
    subparsers = register_subparsers(subparsers, "automation-plots", "Automation plots management", short="ap")

    register_parser(subparsers, "list", _on_list, "List automation plots", short="l")

    _add_plot_action(subparsers, "get", _on_get, "Show all information about the specified automation plot")

    _add_plot_check_action(subparsers, "get-check", _on_get_check,
                           "Show specified check information in specified automation plot", short="gc")

    _add_plot_action(subparsers, "remove", _on_remove, "Remove automation plot", short="r", with_reason=True)

    parser = _add_plot_action(subparsers, "add", _on_add, "Add new automation plot", short="a", with_reason=True)
    parser.add_argument("name", help="name that will identify the automation plot in UI")
    parser.add_argument("-o", "--owners", metavar="LOGINS", help="a comma-separated list of automation-plot owners")

    parser = _add_plot_check_action(subparsers, "add-check", _on_add_check,
                                    "Add check to automation plot", short="ac", with_reason=True)
    _init_check_edit_arguments(parser)

    parser = _add_plot_check_action(subparsers, "edit-check", _on_modify_check,
                                    "Edit automation plot check", short="ec", with_reason=True)
    _init_check_edit_arguments(parser)

    parser = _add_plot_check_action(subparsers, "remove-check", _on_remove_check,
                                    "Edit automation plot check", short="rc", with_reason=True)
    _init_check_edit_arguments(parser)

    _init_enable_check_action(subparsers)
    _init_disable_check_action(subparsers)


def _init_check_edit_arguments(parser):
    parser.add_argument("--reboot", type=str2bool, nargs='?', default=None,
                        help="Reboot failure action [yes|no|y|n|true|false|t|f|1|0]")
    parser.add_argument("--redeploy", type=str2bool, nargs='?', default=None,
                        help="Reboot failure action [yes|no|y|n|true|false|t|f|1|0]")
    parser.add_argument("--wait", type=str2bool, nargs='?', default=None,
                        help="Wait check action [yes|no|y|n|true|false|t|f|1|0]")
    parser.add_argument("--report", type=str2bool, nargs='?', default=None,
                        help="Report failure action [yes|no|y|n|true|false|t|f|1|0]")


def _drop_unwanted_fields_from_checks(checks):
    for check in checks:
        for flag in ("reboot", "redeploy", "wait", "report_failure", "enabled"):
            check[flag] = check.get(flag, False) or False
        check.pop("start_time", None)
    return checks


def add_automation_plot_check(plot, check_name,
                              wait=None, reboot=None, redeploy=None, report_failure=None):
    for check in plot["checks"]:
        if check["name"] == check_name:
            raise Error("Check {} already exists in plot {}".format(check_name, plot['id']))
    plot["checks"].append({
        "name": check_name
    })
    modify_automation_plot_check(plot, check_name,
                                 wait=wait, reboot=reboot, redeploy=redeploy, report_failure=report_failure)


def modify_automation_plot_check(plot, check_name,
                                 wait=None, reboot=None, redeploy=None, report_failure=None):
    report_failure = report_failure or redeploy or reboot
    for check in plot["checks"]:
        if check["name"] == check_name:

            check.update({
                            "wait": wait or False,
                            "reboot": reboot or False,
                            "redeploy": redeploy or False,
                            "report_failure": report_failure or False
            })
            return
    raise Error("Check {} not found in plot {}".format(check_name, plot['id']))


def remove_automation_plot_check(plot, check_name):
    new_checks = []
    check_found = False
    for check in plot["checks"]:
        if check["name"] == check_name:
            check_found = True
        else:
            new_checks.append(check)
    plot["checks"] = new_checks
    if not check_found:
        raise Error("Check {} not found in plot {}".format(check_name, plot['id']))


def extract_check(plot, check_name):
    for check in plot["checks"]:
        if check["name"] == check_name:
            return check
    raise Error("Check {} not found in plot {}".format(check_name, plot['id']))


def _init_enable_check_action(subparsers):
    _add_plot_check_action(subparsers, "enable-check", _on_enable_check,
                           "Enable specified check", with_reason=True)


def _init_disable_check_action(subparsers):
    _add_plot_check_action(subparsers, "disable-check", _on_disable_check,
                           "Disable specified check", with_reason=True)


def _add_plot_action(subparsers, name, handler, description, short=None, with_reason=False):
    parser = register_parser(subparsers, name, handler, description, short=short, with_reason=with_reason)
    parser.add_argument("id", help="automation-plot ID")
    return parser


def _add_plot_check_action(subparsers, name, handler, description, short=None, with_reason=False):
    parser = _add_plot_action(subparsers, name, handler, description, short=short, with_reason=with_reason)
    parser.add_argument("check", help="check name")
    return parser


def get_plot(client, plot_id):
    return client.get_automation_plot(plot_id, AUTOMATION_PLOT_FIELDS)


def _on_add_check(client, args):
    plot = get_plot(client, args.id)
    add_automation_plot_check(plot, args.check,
                              wait=args.wait,
                              reboot=args.reboot,
                              redeploy=args.redeploy,
                              report_failure=args.report)
    client.modify_automation_plot(args.id, checks=_drop_unwanted_fields_from_checks(plot["checks"]))


def _on_modify_check(client, args):
    plot = get_plot(client, args.id)
    modify_automation_plot_check(plot, args.check,
                                 wait=args.wait,
                                 reboot=args.reboot,
                                 redeploy=args.redeploy,
                                 report_failure=args.report)
    client.modify_automation_plot(args.id, checks=_drop_unwanted_fields_from_checks(plot["checks"]))


def _on_remove_check(client, args):
    plot = get_plot(client, args.id)
    remove_automation_plot_check(plot, args.check)
    client.modify_automation_plot(args.id, checks=_drop_unwanted_fields_from_checks(plot["checks"]))


def _on_add(client, args):
    client.add_automation_plot(args.id, args.name, parse_arg_list(args.owners), reason=args.reason)


def _on_remove(client, args):
    client.remove_automation_plot(args.id, reason=args.reason)


@render_result
def _on_get(client, args):
    plot = get_plot(client, args.id)
    return {"automation-plot": plot}


@render_result
def _on_get_check(client, args):
    plot = get_plot(client, args.id)
    check = extract_check(plot, args.check)
    return {args.check: check}


@render_result
def _on_list(client, args):
    plots = client.get_automation_plots(AUTOMATION_PLOT_LIST_FIELDS)
    return {"automation-plots": plots}


def _on_disable_check(client, args):
    client.disable_check(id=args.id, check_name=args.check, reason=args.reason)


def _on_enable_check(client, args):
    client.enable_check(id=args.id, check_name=args.check, reason=args.reason)
