"""Display host health data"""
from collections import OrderedDict
from functools import partial

from walle_api.constants import CheckStatus, CheckType

from walle_cli.common import register_parser, register_subparsers, render_yaml_unsorted, format_choices, render_result


def init(subparsers):
    subparsers = register_subparsers(subparsers, "health-checks", "Health checks management", short="hc")
    _init_list(subparsers)


def _init_list(subparsers):
    parser = register_parser(subparsers, "list", _on_list, "Show health checks")
    parser.add_argument("--fqdn", help="filter by FQDN")
    parser.add_argument("-t", "--type", choices=format_choices(CheckType.ALL), help="filter by health check type")
    parser.add_argument("-s", "--status", choices=format_choices(CheckStatus.ALL), help="filter by status")
    parser.add_argument("--timestamp", help="filter by timestamp (exact)")
    limit_group = parser.add_mutually_exclusive_group()
    limit_group.add_argument("-l", "--limit", type=int, default=100,
                             help="maximum number of returned health checks (default 100)")
    limit_group.add_argument("-a", "--show-all", action="store_true", help="show all health checks")


def _get_dict_in_key_order(unordered_dict, key_order):
    ordered_dict = OrderedDict()
    for key in key_order:
        ordered_dict[key] = unordered_dict[key]
    return ordered_dict


def _display_health_checks(health_checks):
    key_order = ["id", "fqdn", "type", "status", "status_mtime", "timestamp", "metadata"]
    make_ordered = partial(_get_dict_in_key_order, key_order=key_order)

    return {"checks": [make_ordered(check) for check in health_checks]}


@render_result(renderer=render_yaml_unsorted)
def _on_list(client, args):
    params = {
        "fqdn": args.fqdn,
        "type": args.type,
        "status": args.status,
        "timestamp": args.timestamp,
        "limit": args.limit if not args.show_all else None,
    }
    return _display_health_checks(client.iter_health_checks(**params))
