"""Host group management."""

import walle_cli.hosts
import walle_cli.task_processing
from object_validator import DictScheme, List, String, Integer
from walle_cli.common import Error, register_parser, register_subparsers
from walle_cli.model import Model, ObjectDoesntExistError, ObjectAlreadyExistsError


class HostGroup(Model):
    _collection_name = "host-groups"
    _id_field = "name"
    _scheme = DictScheme({
        "name": String(min_length=1),
        "hosts": List(String(min_length=1)),
        "max_fail_percent": Integer(min=0, max=100),
        "process_queue_size": Integer(min=1),
        "task": DictScheme({
            "type": String(choices=walle_cli.task_processing.TaskType.ALL),
            "hosts": List(String(min_length=1)),
            "params": DictScheme({}, ignore_unknown=True, optional=True),
            "in_process": List(String(min_length=1), optional=True),
            "completed": List(String(min_length=1), optional=True),
            "failed": List(String(min_length=1), optional=True),
        }, optional=True),
    })


def init(subparsers):
    subparsers = register_subparsers(subparsers, "host-groups", "Host group management (saved locally in ~/.wall-e)", short="g")

    register_parser(subparsers, "list", _on_list, "List all registered host groups")

    _add_group_action(subparsers, "show", _on_show, "Show the specified host group")
    _add_group_action(subparsers, "process", _on_process, "Process currently assigned task")
    _add_group_action(subparsers, "cancel", _on_cancel, "Cancels currently assigned task", short="k")

    parser = walle_cli.hosts.add_host_action(subparsers, "create", _on_create, "Create a new host group",
                                             allow_invs=False, with_groups=False)
    parser.add_argument("-e", "--max-fail-percent", type=int, default=5,
                        help="Maximum failure percent when processing host tasks (default is 5)")
    parser.add_argument("-m", "--process-queue-size", type=int, default=20,
                        help="Process queue size for long running tasks (reboot/redeploy) (default is 20)")
    parser.add_argument("-n", "--name", dest="group_name", required=True, help="Group name")

    _add_group_action(subparsers, "remove", _on_remove, "Remove a host group", short="d")


def get_group(name, locked=False):
    try:
        return HostGroup.get_object(name, locked=locked)
    except ObjectDoesntExistError:
        raise Error("Host group '{}' doesn't exist.", name)


def _on_list(client, args):
    names = HostGroup.list_objects()
    if not names:
        return

    print("Registered host groups:")
    for name in names:
        print("*", name)


def _on_show(client, args):
    group = get_group(_validate_group_name(args.name))
    print("Host group '{}' ({}):\n".format(group["name"], group.get_path()))
    print(group.to_yaml().strip())


def _on_process(client, args):
    group = get_group(_validate_group_name(args.name), locked=True)

    try:
        walle_cli.task_processing.process_task(client, group)
    finally:
        group.unlock()


def _on_cancel(client, args):
    group = get_group(_validate_group_name(args.name), locked=True)

    try:
        if "task" in group:
            del group["task"]
            group.save()
    finally:
        group.unlock()


def _on_create(client, args):
    name = _validate_group_name(args.group_name)

    if args.process_queue_size < 1:
        raise Error("Invalid process queue size.")

    if args.max_fail_percent < 0 or args.max_fail_percent > 100:
        raise Error("Invalid maximum failure percent value.")

    hosts = walle_cli.hosts.get_host_name_args(args, with_host_groups=False)

    group = HostGroup({
        "name": name,
        "hosts": hosts,
        "process_queue_size": args.process_queue_size,
        "max_fail_percent": args.max_fail_percent
    })

    try:
        group.save(force_creation=True)
    except ObjectAlreadyExistsError:
        raise Error("Unable to create host group '{}': group is already exists.", group["name"])


def _on_remove(client, args):
    name = _validate_group_name(args.name)

    try:
        group = HostGroup.get_object(name, locked=True)
    except ObjectDoesntExistError:
        raise Error("Unable to remove host group '{}': it doesn't exist.", name)

    try:
        group.remove()
    except:
        group.unlock()
        raise


def _add_group_action(subparsers, name, handler, description, short=None):
    parser = register_parser(subparsers, name, handler, description, short=short)
    parser.add_argument("name", help="Group name")
    return parser


def _validate_group_name(name):
    if not HostGroup.id_regex.search(name):
        raise Error("Invalid group name.")

    return name
