"""Host management."""

import re
import sys
from functools import partial
from operator import itemgetter

from walle_api.client import WalleClient
from walle_api.constants import Provisioner, HostStatus, HostOperationState, Restriction, SshOptions, DeployNetwork, \
    CmsTaskAction

from walle_cli import task_processing
from walle_cli import host_groups
from walle_cli.common import Error, LogicalError, TableView, Column, register_subparsers, register_parser, \
    parse_arg_list, format_choices, format_time, get_supported_time_formats, parse_time,  question_user, \
    process_args, unique_list, drop_none, render_result, print_object

_SYNTAX_PLAIN = "plain"
_SYNTAX_BLINOV = "blinov"
_SYNTAXES = [_SYNTAX_PLAIN, _SYNTAX_BLINOV]
_TABLE_VIEW = TableView(
    default_columns=(
        Column("uuid", "UUID"),
        Column("inv", "Inv.num"),
        Column("name", "FQDN"),
        Column("state", "State"),
        Column("status", "Status"),
        Column("ticket", "Ticket"),
        Column("task_status", "Task status", align=Column.ALIGN_CENTER, fields=(
            "status", "task.status", "task.deploy.provisioner_status", "provisioner")),
        Column("task_error", "Task error", align=Column.ALIGN_LEFT, fields=("task.error",)),
        Column("extra_vlans", "Extra vlans")
    ),
    other_columns=(
        Column("project", "Project", align=Column.ALIGN_CENTER),
        Column("tags", "Tags", align=Column.ALIGN_LEFT),
        Column("provisioner", "Provisioner", align=Column.ALIGN_CENTER,
               fields=("provisioner", "custom_deploy_configuration")),
        Column("config", "Config", fields=("config", "custom_deploy_configuration")),
        Column("deploy_config_policy", "Deploy config policy",
               align=Column.ALIGN_CENTER,
               fields=("deploy_config_policy", "custom_deploy_configuration")),
        Column("deploy_tags", "Deploy tags", align=Column.ALIGN_CENTER),
        Column("deploy_network", "Deploy network", align=Column.ALIGN_CENTER),
        Column("custom_deploy_configuration", "Custom deploy config", align=Column.ALIGN_CENTER),
        Column("restrictions", "Restrictions", align=Column.ALIGN_CENTER),
        Column("task_owner", "Task owner", align=Column.ALIGN_CENTER, fields=("task.owner",)),
        Column("task_status_message", "Task status message", align=Column.ALIGN_LEFT, fields=("task.status_message",)),
        Column("task_log_id", "Task log id", align=Column.ALIGN_LEFT, fields=("task.audit_log_id",)),
        Column("health", "Health", align=Column.ALIGN_CENTER, fields=("health.status", "health.reasons")),

        Column("status_reason", "Status reason", align=Column.ALIGN_CENTER),
        Column("status_author", "Status author", align=Column.ALIGN_CENTER),
        Column("status_time", "Status time", align=Column.ALIGN_CENTER),
        Column("status_audit_log_id", "Status log id", align=Column.ALIGN_CENTER),
        Column("state_reason", "State reason", align=Column.ALIGN_CENTER),
        Column("state_author", "State author", align=Column.ALIGN_CENTER),
        Column("state_time", "State time", align=Column.ALIGN_CENTER),
        Column("state_audit_log_id", "State log id", align=Column.ALIGN_CENTER),
        Column("state_timeout", "State timeout", align=Column.ALIGN_CENTER),

        Column("messages", "Messages", align=Column.ALIGN_CENTER),
        Column("juggler_aggregate_name", "Aggregate", align=Column.ALIGN_CENTER),

        #IPs
        Column("ips", "IPs", align=Column.ALIGN_LEFT),
        Column("ips_time", "IPs time"),

        # MACs
        Column("ipmi_mac", "IPMI MAC", align=Column.ALIGN_CENTER),
        Column("macs", "MACs", align=Column.ALIGN_LEFT),
        Column("active_mac", "Active MAC", align=Column.ALIGN_CENTER),
        Column("active_mac_source", "Active MAC source", align=Column.ALIGN_CENTER),
        Column("active_mac_time", "Active MAC time"),
        # Network location
        Column("switch", "Switch", fields=("location.switch",)),
        Column("port", "Port", fields=("location.port",)),
        Column("netmap_time", "Netmap time", fields=("network_timestamp",)),
        Column("netmap_source", "Netmap source", fields=("location.network_source",)),
        # Physical location
        Column("geo_location", "Geo location", align=Column.ALIGN_LEFT,
               fields=["location.%s" % f for f in ["country", "city", "datacenter", "queue", "rack"]]),
        Column("geo_time", "Geo time", fields=("location.physical_timestamp",)),

        Column("agent_version", "Wall-E.agent"),
        Column("state_expire", "State Expire"),
        Column("platform", "Platform", align=Column.ALIGN_CENTER),
        Column("operation_state", "Operation State", align=Column.ALIGN_CENTER),
        Column("scenario_id", "Scenario ID", align=Column.ALIGN_CENTER)
    ),
)


def init(subparsers):
    subparsers = register_subparsers(subparsers, "hosts", "Host management")

    _init_adding(subparsers)
    _init_listing(subparsers)
    _init_modification(subparsers)
    _init_state_forcing(subparsers)
    _init_operations(subparsers)


def _init_listing(subparsers):
    list_parser = register_parser(subparsers, "list", _on_list, "List all registered hosts")
    list_only_parser = add_host_action(subparsers, "list-only", _on_list_only, "List only specified hosts",
                                       short="lo", with_ignore_maintenance=False)

    list_parser.add_argument("-g", "--group", help="host group to work on "
                             "(see: walle-e host-groups --help)")

    for parser in [list_parser, list_only_parser]:
        parser.add_argument("-n", "--name", help="filter by name substring")
        parser.add_argument("-S", "--state", help="filter by states split by comma")
        parser.add_argument("-s", "--status", help="filter by statuses split by comma")
        parser.add_argument("--switch", help="filter by switch names split by comma")
        parser.add_argument("--port", help="filter by port names split by comma")
        parser.add_argument("-H", "--health", help="filter by host health status: "
                            "ok - healthy host, failure - any failure, no - no health status info, "
                            "$check_type - all $check_type failure reasons, "
                            "$check_type-$reason - the specific failure reason.")
        parser.add_argument("-dc", "--deploy-config-policy", help="filter by deploy configuration policy")

        parser.add_argument("-p", "--project", help="filter by project")
        parser.add_argument("-t", "--tags", help="filter by project's tags")
        parser.add_argument("-G", "--geo", help="filter by physical location")
        parser.add_argument("-i", "--issuer", help="filter by task owner (don't forget @ after user name)")
        parser.add_argument("-P", "--provisioner", help="filter by provisioner")
        parser.add_argument("-c", "--config", help="filter by config substring")
        parser.add_argument("-r", "--restrictions", help="filter by restrictions")
        parser.add_argument("-o", "--scenario", help="filter by scenario_id")

        parser.add_argument("-C", "--columns", help="a comma-separated list of columns to output ({})".format(
                            ", ".join(_TABLE_VIEW.get_column_ids())))

    list_parser.add_argument("-L", "--limit", type=int, default=100,
                             help="maximum number of returned hosts (default is 100)")
    list_only_parser.add_argument("-L", "--limit", type=int,
                                  help="maximum number of returned hosts (default is unlimited)")

    get_parser = add_host_action(subparsers, "get", _on_get, "Show all information about the specified host", short="g",
                                 multi=False, with_ignore_maintenance=False)
    get_parser.add_argument("-c", "--current-configuration", action="store_true",
                            help="show current host configuration with best effort up-to-date status")
    get_parser.add_argument("-H", "--health-data", action="store_true", help="show host health checks")
    get_parser.add_argument("-p", "--power-status", action="store_true", help="show current power status for host")

    get_deploy_log_parser = add_host_action(subparsers, "deploy-log", _on_get_deploy_log, "Get host deploy log",
                                            short="dl", multi=False, with_ignore_maintenance=False)
    get_deploy_log_parser.add_argument("-p", "--provisioner", choices=format_choices(Provisioner.ALL),
                                       help="provisioner")
    get_deploy_log_parser.add_argument("-b", "--tail-bytes", "-s", "--size", metavar="size", type=int, default=100000,
                                       help="return only the specified number of bytes from the tail "
                                            "(default is 100000). If provisioner doesn't support this feature "
                                            "the whole log will be returned.")

    add_host_action(subparsers, "profile-log", _on_get_profile_log, "Get host profile log",
                    short="pl", multi=False, with_ignore_maintenance=False)


def _init_adding(subparsers):
    parser = register_parser(subparsers, "add", _on_add, "Add hosts to the system", with_reason=True)
    parser.add_argument("project", help="project ID")
    _add_host_action_argument(parser, with_ignore_maintenance=False)
    _add_new_task_arguments(parser)
    _add_host_deploy_config_arguments(parser)

    parser.add_argument("-r", "--restrictions",
                        help="a comma-separated list of restrictions ({})".format(", ".join(Restriction.ALL)))
    parser.add_argument("-S", "--state", help="state to set for the host")
    parser.add_argument("-s", "--status", help="status to set for the host")
    parser.add_argument("-d", "--dns", action="store_true", default=None, help="check dns records for the host")
    parser.add_argument("-i", "--instant", action="store_true",
                        help="instant operation, don't create task, ask CMS, etc")

    # Maintenance parameters
    parser.add_argument("-po", "--power-off", action="store_true", help="power off the host")
    parser.add_argument("-st", "--ticket-key", help="ticket key to attach when forcing host status")
    parser.add_argument("--ttl", help="status TTL in one of the following formats: " + get_supported_time_formats())
    parser.add_argument("-ts", "--timeout-status", choices=format_choices([HostStatus.READY, HostStatus.DEAD]),
                        help="status the host will be switched to on TTL expiration")
    parser.add_argument("-os", "--operation-state", choices=format_choices(HostOperationState.ALL),
                        help="operation state the host will be switched to in maintenance")
    parser.add_argument("-ct", "--cms-task-action", choices=format_choices(CmsTaskAction.ALL),
                        help="CMS task action for switching to maintenance")


def _init_modification(subparsers):
    parser = add_host_action(subparsers, "modify", _on_modify, "Modify hosts", with_reason=True)
    parser.add_argument("-r", "--restrictions", required=True,
                        help="a comma-separated list of restrictions ({})".format(", ".join(Restriction.ALL)))

    parser = register_subparsers(subparsers, "deploy-config", "Configure host's deploy config", short="dc")
    add_host_action(parser, "remove", _on_clear_deploy_config, "Remove deploy config", with_reason=True)
    set_deploy_config_parser = add_host_action(parser, "set", _on_set_deploy_config, "Set deploy config",
                                               with_reason=True)
    _add_host_deploy_config_arguments(set_deploy_config_parser, required=True)

    subparsers = register_subparsers(subparsers, "extra-vlans", "Configure host's extra VLANs")
    add_host_action(subparsers, "get", _on_get_extra_vlans, "Show configured extra VLANs",
                    multi=False, with_ignore_maintenance=False)

    for action in WalleClient.get_api_actions():
        parser = add_host_action(subparsers, action, _get_on_modify_extra_vlans_handler(action),
                                 "{} extra VLANs".format(action.capitalize()), with_reason=True)
        parser.add_argument("vlans", help="a comma-separated list of VLANs to {}".format(action))


def _init_operations(subparsers):
    parser = add_host_action(subparsers, "prepare", _on_prepare, "Prepare hosts", short="c", with_reason=True)
    parser.add_argument("--skip-profile", default=None, action="store_true", help="don't profile the host")
    parser.add_argument("--profile", help="Einstellung profile to assign to the host")
    parser.add_argument("--profile-tags", help="Einstellung tags to assign to the host before profiling")
    parser.add_argument("--provisioner", choices=format_choices(Provisioner.ALL), help="provisioner")
    parser.add_argument("-c", "--config", help="deploy config name")
    parser.add_argument("-dcp", "--deploy-config-policy", help="Policy to (optionally) alter deploy config")
    parser.add_argument("--deploy-tags", help="Deploy tags to assign to the host before deploying")
    parser.add_argument("--deploy-network", choices=DeployNetwork.ALL, help="deploy host in project or service network")
    parser.add_argument("--keep-name", action="store_true", help="keep host's current name")

    parser.add_argument("-r", "--restrictions",
                        help="a comma-separated list of restrictions ({})".format(", ".join(Restriction.ALL)))
    _add_new_task_arguments(parser)

    parser = add_host_action(subparsers, "power-on", _on_power_on, "Power on hosts", short="o", with_reason=True)
    _add_new_task_arguments(parser, with_cms=False)

    parser = add_host_action(subparsers, "power-off", _on_power_off, "Power off hosts", short="f", with_reason=True)
    _add_new_task_arguments(parser, with_check=False, with_auto_healing=False)
    parser.add_argument("-o", "--operation-state", required=True, choices=format_choices(HostOperationState.ALL),
                        help="operation state the host will be switched to after power off")
    parser.add_argument("-t", "--ticket-key", required=True, help="ticket key to attach when forcing host status")
    parser.add_argument("--ttl", help="status TTL in one of the following formats: " + get_supported_time_formats())
    parser.add_argument("-s", "--timeout-status", choices=format_choices([HostStatus.READY, HostStatus.DEAD]),
                        help="status the host will be switched to on TTL expiration")
    parser.add_argument("-c", "--cms-task-action", choices=format_choices(CmsTaskAction.ALL),
                        help="CMS task action for switching to maintenance")

    parser = add_host_action(subparsers, "reboot", _on_reboot, "Reboot hosts", with_reason=True)
    parser.add_argument("--ssh", choices=format_choices(SshOptions.ALL),
                        help="Allow or forbid the use of ssh for the operation (defaults to forbid).")
    _add_new_task_arguments(parser)

    parser = add_host_action(subparsers, "check-dns", _on_check_dns, "Ckeck and fix host DNS records",
                             short="dns", with_reason=True)
    _add_new_task_arguments(parser, with_cms=False)

    parser = add_host_action(subparsers, "profile", _on_profile, "Profile hosts", with_reason=True)
    parser.add_argument("-p", "--profile", help="Einstellung profile name")
    parser.add_argument("-t", "--profile-tags", help="a comma-separated list of Einstellung tags")
    parser.add_argument("-r", "--redeploy", default=None, action="store_true", help="redeploy host after profile")
    parser.add_argument("-P", "--provisioner", choices=format_choices(Provisioner.ALL), help="deploy provisioner")
    parser.add_argument("-c", "--config", help="deploy config name")
    parser.add_argument("-dcp", "--deploy-config-policy", help="Policy to (optionally) alter deploy config")
    parser.add_argument("-T", "--deploy-tags", help="Deploy tags to assign to the host before deploying")
    parser.add_argument("-n", "--deploy-network", choices=DeployNetwork.ALL,
                        help="deploy host in project or service network")

    _add_new_task_arguments(parser)

    parser = add_host_action(subparsers, "redeploy", _on_redeploy, "Redeploy hosts", short="i", with_reason=True)
    parser.add_argument("-p", "--provisioner", choices=format_choices(Provisioner.ALL), help="provisioner")
    parser.add_argument("-c", "--config", help="deploy config name")
    parser.add_argument("-dcp", "--deploy-config-policy", help="Policy to (optionally) alter deploy config")
    parser.add_argument("-T", "--deploy-tags", help="a comma-separated list of Einstellung tags")
    parser.add_argument("-n", "--deploy-network", choices=DeployNetwork.ALL,
                        help="deploy host in project or service network")
    _add_new_task_arguments(parser)

    parser = add_host_action(subparsers, "switch-vlans", _on_switch_vlans, "Switch VLANs", short="v", with_reason=True)
    parser.add_argument("--vlans", help="A comma-separated list of VLANs to assign to the port. "
                                        "The first one will be set as native VLAN ID.")
    parser.add_argument("--network-target", help="Target network. One of: "
                                                 "service (for VLAN 542), "
                                                 "parking (for VLAN 999), "
                                                 "project (for vlans from vlan scheme).")
    add_host_action(subparsers, "apply-vlans", _on_switch_vlans, "Apply VLANs to RackTables",
                    short="av", with_reason=True)

    parser = add_host_action(subparsers, "switch-project", _on_switch_project, "Switch project", short="t", with_reason=True)
    parser.add_argument("project", help="project ID to switch the host into")
    parser.add_argument("--release", default=False, action="store_true",
                        help="release host before switching to the project")
    parser.add_argument("--no-erase-disks", action="store_false", dest="erase_disks",
                        help="do not try to erase host's disks when releasing the host")
    parser.add_argument("--restrictions",
                        help="a comma-separated list of restrictions to assign to the host after switching ({})".format(
                             ", ".join(Restriction.ALL)))
    parser.add_argument("-f", "--force", action="store_true",
                        help="force switching without checking the projects for compatibility")
    _add_new_task_arguments(parser, with_check=False, with_auto_healing=False)

    parser = add_host_action(subparsers, "release-host", _on_release_host, "Release host", short="rh",
                             with_reason=True)
    parser.add_argument("--no-erase-disks", action="store_false", dest="erase_disks",
                        help="Try to not erase host's disks when releasing the host")
    _add_new_task_arguments(parser, with_check=False, with_auto_healing=False)

    parser = add_host_action(subparsers, "remove", _on_remove, "Remove hosts from the system",
                             short="d", with_reason=True)
    parser.add_argument("-l", "--lui", default=False, action="store_true", help="also remove from LUI")
    parser.add_argument("-i", "--instant", default=False, action="store_true",
                        help="instant operation, don't create task, don't aks CMS, etc")
    _add_new_task_arguments(parser, with_check=False, with_auto_healing=False)

    _init_fqdn_deinvalidation_operation(subparsers)

    parser = add_host_action(subparsers, "handle-failure", _on_handle_failure, "Handle host failure", with_reason=True)
    parser.add_argument("--ignore-cms", action="store_true", help="don't acquire permission from CMS")


def _init_state_forcing(subparsers):
    parser = add_host_action(subparsers, "set-maintenance", _on_set_maintenance,
                             "Set host maintenance state (cancels any processing tasks). "
                             "If host has ticket and no ttl it will leave maintenance on ticket resolve.", short="sm",
                             with_reason=True, with_ignore_maintenance=False)
    parser.add_argument("-p", "--power-off", action="store_true", help="power off the host")
    parser.add_argument("-t", "--ticket-key", required=True, help="ticket key to attach when forcing host status")
    parser.add_argument("--ttl", help="status TTL in one of the following formats: " + get_supported_time_formats())
    parser.add_argument("-s", "--timeout-status", choices=format_choices([HostStatus.READY, HostStatus.DEAD]),
                        help="status the host will be switched to on TTL expiration")
    parser.add_argument("-o", "--operation-state", choices=format_choices(HostOperationState.ALL),
                        help="operation state the host will be switched to in maintenance")
    parser.add_argument("-c", "--cms-task-action", choices=format_choices(CmsTaskAction.ALL),
                        help="CMS task action for switching to maintenance")
    _add_new_task_arguments(parser, with_check=False)

    parser = add_host_action(subparsers, "change-maintenance", _on_change_maintenance,
                             "Change host maintenance parameters. "
                             "If host has ticket and no ttl it will leave maintenance on ticket resolve.", short="cm",
                             with_reason=True, with_ignore_maintenance=False)
    parser.add_argument("-t", "--ticket-key", help="ticket key to attach when forcing host status")
    parser.add_argument("--ttl", help="status TTL in one of the following formats: " + get_supported_time_formats())
    parser.add_argument("-s", "--timeout-status", choices=format_choices([HostStatus.READY, HostStatus.DEAD]),
                        help="status the host will be switched to on TTL expiration")
    parser.add_argument("-o", "--operation-state", choices=format_choices(HostOperationState.ALL),
                        help="operation state the host will be switched to in maintenance")
    parser.add_argument("--remove-ttl", action="store_true",
                        help="remove maintenance timeout in favour of ticket-based state change")

    parser = add_host_action(subparsers, "set-assigned", _on_set_assigned,
                             "Set host assigned state", short="sa", with_reason=True)
    parser.add_argument("-p", "--power-on", action="store_true", help="power on the host")
    parser.add_argument("-s", "--status", choices=format_choices([HostStatus.READY, HostStatus.DEAD]),
                        help="switch host to specified status")
    _add_new_task_arguments(parser, with_cms=False)

    parser = add_host_action(subparsers, "force-status", _on_force_status,
                             "Force host status (cancels any processing tasks)"
                             "If host in maintenance has ticket and no ttl "
                             "it will leave maintenance on ticket resolve.",
                             short="s", with_reason=True)
    parser.add_argument("status", choices=format_choices([HostStatus.READY, HostStatus.MANUAL, HostStatus.DEAD]),
                        help="status to force")
    parser.add_argument("-t", "--ticket-key", help="ticket key to attach when forcing host status")
    parser.add_argument("--ttl", help="status TTL in one of the following formats: " + get_supported_time_formats())
    parser.add_argument("-s", "--timeout-status", choices=format_choices(
        [HostStatus.READY, HostStatus.DEAD]),
                        help="status the host will be switched to on TTL expiration")

    add_host_action(subparsers, "cancel-task", _on_cancel_task,
                    "Cancel currently processing task. Sets default status for current state: "
                    "MANUAL for MAINTENANCE, READY for FREE and ASSIGNED",
                    short="ct", with_reason=True)


def add_host_action(subparsers, name, handler, description, short=None, allow_invs=True, multi=True,
                    with_ignore_maintenance=True, with_groups=True, with_reason=False):
    parser = register_parser(subparsers, name, handler, description,
                             short=short, with_reason=with_reason)
    _add_host_action_argument(parser, allow_invs=allow_invs, multi=multi, with_groups=with_groups,
                              with_ignore_maintenance=with_ignore_maintenance)
    return parser


def _add_host_action_argument(parser, allow_invs=True, multi=True, with_groups=True, with_ignore_maintenance=True):
    kwargs = {}
    if multi:
        kwargs.update(nargs="*")

    if allow_invs:
        arg_name = "id"
        kwargs.update(help="host inventory number or FQDN")
    else:
        arg_name = "name"
        kwargs.update(help="FQDN")

    parser.add_argument(arg_name, **kwargs)

    if multi:
        if with_groups:
            parser.add_argument("-g", "--group", help="host group to work on "
                                "(see: walle-e host-groups --help)")
            parser.add_argument("-a", "--all", default=False, action="store_true",
                                help="operate on all hosts of the specified host group")

        parser.add_argument("--syntax", default=_SYNTAX_PLAIN, choices=format_choices(_SYNTAXES),
                            help="host list syntax to use: plain or blinov "
                                 "(blinov requires skynet installed)")

    if with_ignore_maintenance:
        parser.add_argument("--ignore-maintenance", default=None, action="store_true",
                            help="forcefully submit operation ignoring host maintenance status")


def _host_action_kwargs(args, with_ignore_maintenance=True):
    return drop_none({
        "reason": args.reason,
        "ignore_maintenance": args.ignore_maintenance if with_ignore_maintenance else None
    })


def _task_kwargs(args, with_cms=True, with_check=True, with_auto_healing=True):
    return drop_none({
        "ignore_cms": True if with_cms and args.ignore_cms else None,
        "disable_admin_requests": True if args.no_admin_requests else None,
        "check": False if with_check and args.no_check else None,
        "with_auto_healing": True if with_check and with_auto_healing and args.with_auto_healing else None,
    })


def get_host_name_args(args, host_groups=True):
    invs, names = _get_host_args(args, args.name, split=True, with_host_groups=host_groups)

    if invs:
        raise Error("Invalid host names: {}.", ", ".join(str(inv) for inv in invs))

    return names


def _add_new_task_arguments(parser, with_cms=True, with_check=True, with_auto_healing=True):
    if with_cms:
        parser.add_argument("--ignore-cms", action="store_true", help="don't acquire permission from CMS")

    parser.add_argument("--no-admin-requests", action="store_true",
                        help="don't issue any admin requests if something is broken - just fail the task")

    if with_check:
        parser.add_argument("--no-check", action="store_true",
                            help="don't check host health status after task completion")

        if with_auto_healing:
            parser.add_argument("--with-auto-healing", action="store_true",
                                help="try to automatically repair host if task fails")


def _add_host_deploy_config_arguments(parser, required=False):
    parser.add_argument("-p", "--provisioner", choices=format_choices(Provisioner.ALL), help="provisioner")
    parser.add_argument("-c", "--config", required=required, help="deploy config name")
    parser.add_argument("-dcp", "--deploy-config-policy", help="Policy to (optionally) alter deploy config")
    parser.add_argument("-T", "--deploy-tags", help="a comma-separated list of Einstellung tags")
    parser.add_argument("-n", "--deploy-network", choices=DeployNetwork.ALL,
                        help="deploy host in project or service network")


def _display_geo_location(location):
    location_components = [location.get(f) for f in ["country", "city", "datacenter", "queue", "rack"]]
    if not any(location_components):
        return "-"
    return "|".join(str(x) or "-" for x in location_components)


@render_result
def _on_get(client, args):
    host = client.get_host(args.id, resolve_deploy_configuration=True, fields=_TABLE_VIEW.get_column_fields() + [
        "status_timeout", "status_audit_log_id"])

    host_name = host.get("name")
    host.setdefault("health", None)
    host.setdefault("restrictions", [])

    status_timeout_time = host.get("status_timeout", {}).get("time")
    if status_timeout_time is not None:
        host["status_timeout"]["time"] = format_time(status_timeout_time)

    if "active_mac_time" in host:
        host["active_mac_time"] = format_time(host["active_mac_time"])

    if "ips_time" in host:
        host["ips_time"] = format_time(host["ips_time"])

    if "network_timestamp" in host:
        host["netmap_time"] = format_time(host.pop("network_timestamp"))

    location = host.get("location")
    if location is not None:

        if "network_source" in location:
            location["netmap_source"] = location.pop("network_source")

        if "physical_timestamp" in location:
            location["geo_time"] = format_time(location.pop("physical_timestamp"))

        location["geo_location"] = _display_geo_location(location)

    del host["name" if host_name == args.id else "inv"]

    if "status_audit_log_id" in host:
        host["status_reason"] = client.get_event_reason(host["status_audit_log_id"], with_error=True)
        del host["status_audit_log_id"]

    if args.current_configuration:
        host["configuration"] = client.get_host_configuration(args.id)

    if args.power_status:
        host["power_status"] = client.get_power_status(args.id)

    if host_name is not None:
        if args.health_data:
            host["health_data"] = list(client.iter_health_checks(fqdn=host_name))

    return {"host": host}


def _on_list(client, args):
    _list_hosts(client, args, names=None if args.group is None else _get_group_hosts(args.group))


def _on_list_only(client, args):
    invs, names = _get_host_args(args, args.id, split=True)
    _list_hosts(client, args, invs=invs, names=names)


def _list_hosts(client, args, invs=None, names=None):
    columns, fields = _TABLE_VIEW.parse_column_list(args.columns)

    options = dict(invs=invs, names=names,
                   project=args.project, tags=args.tags, name=args.name,
                   state=args.state, status=args.status, health=args.health, task_owner=args.issuer,
                   provisioner=args.provisioner, config=args.config,
                   restrictions=args.restrictions, physical_location=args.geo, switch=args.switch, port=args.port,
                   resolve_deploy_configuration=True, scenario_id=args.scenario, fields=fields)

    limit = args.limit

    if limit is None or limit > client.MAX_PAGE_SIZE:
        hosts = list(client.iter_hosts(limit=limit, **options))

        if limit is None or len(hosts) < limit:
            total = len(hosts)
        else:
            total = max(client.get_hosts(limit=0, **options)["total"], len(hosts))
    else:
        response = client.get_hosts(limit=limit, **options)
        hosts, total = response["result"], response["total"]

    _display_hosts(hosts, columns, args.batch, total=total)


def _on_add(client, args):
    params = _host_action_kwargs(args, with_ignore_maintenance=False)
    params.update(_task_kwargs(args))

    return _process_host_id_args(args, lambda host_id:
        client.add_host(host_id, project=args.project, provisioner=args.provisioner,
                        config=args.config, deploy_config_policy=args.deploy_config_policy,
                        restrictions=parse_arg_list(args.restrictions), state=args.state, status=args.status,
                        deploy_tags=parse_arg_list(args.deploy_tags), deploy_network=args.deploy_network,
                        dns=args.dns, instant=args.instant,
                        maintenance_params=drop_none(_maintenance_params(args)), **params))


def _on_modify(client, args):
    _question_user_on_host_action(args, "modify")
    return _process_host_id_args(args, lambda host_id:
        client.modify_host(host_id, restrictions=parse_arg_list(args.restrictions), **_host_action_kwargs(args)))


@render_result
def _on_get_extra_vlans(client, args):
    extra_vlans = client.get_host(args.id, fields=["extra_vlans"]).get("extra_vlans", [])
    return {"extra_vlans": extra_vlans}


def _get_on_modify_extra_vlans_handler(action):
    def on_modify_extra_vlans(client, args):
        _question_user_on_host_action(args, "modify extra VLANs for")
        return _process_host_id_args(args, lambda host_id:
            client.modify_host_extra_vlans(host_id, action, parse_arg_list(args.vlans, type=int),
                                           **_host_action_kwargs(args)))

    return on_modify_extra_vlans


def _on_set_maintenance(client, args):
    _question_user_on_host_action(args, "set maintenance state for")

    params = _host_action_kwargs(args, with_ignore_maintenance=False)
    params.update(_task_kwargs(args, with_check=False))
    params.update(_maintenance_params(args))

    return _process_host_id_args(args, lambda host_id: client.set_host_maintenance(host_id, **params))


def _on_change_maintenance(client, args):
    _question_user_on_host_action(args, "change maintenance params for")

    return _process_host_id_args(args, lambda host_id:
        client.change_host_maintenance(host_id, ticket_key=args.ticket_key,
                                       timeout_time=parse_time(args.ttl, future_period=True),
                                       timeout_status=args.timeout_status,
                                       operation_state=args.operation_state,
                                       remove_timeout=args.remove_ttl,
                                       **_host_action_kwargs(args, with_ignore_maintenance=False)))


def _on_set_assigned(client, args):
    _question_user_on_host_action(args, "set assigned state for")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_cms=False))

    return _process_host_id_args(args, lambda host_id:
        client.set_host_assigned(host_id, status=args.status, power_on=args.power_on, **params))


def _on_force_status(client, args):
    if args.timeout_status is not None and args.ttl is None:
        raise Error("Timeout status must be specified only with status TTL.")

    _question_user_on_host_action(args, "force status for")
    return _process_host_id_args(args, lambda host_id:
        client.force_host_status(host_id, args.status, timeout_time=parse_time(args.ttl, future_period=True),
                                 timeout_status=args.timeout_status, ticket_key=args.ticket_key,
                                 **_host_action_kwargs(args)))


def _on_cancel_task(client, args):
    _question_user_on_host_action(args, "cancel current task for")
    return _process_host_id_args(args, lambda host_id:
        client.cancel_host_task(host_id, **_host_action_kwargs(args, with_ignore_maintenance=True)))


def _on_switch_vlans(client, args):
    _question_user_on_host_action(args, "switch VLANs for")

    if hasattr(args, "vlans"):
        vlans = parse_arg_list(args.vlans, type=int)
    else:
        vlans = None
    native_vlan = vlans[0] if vlans else None

    network_target = args.network_target if hasattr(args, "network_target") else None

    return _process_host_id_args(args, lambda host_id:
        client.switch_vlans(host_id, vlans=vlans, native_vlan=native_vlan, network_target=network_target,
                            **_host_action_kwargs(args)))


def _on_switch_project(client, args):
    action = "switch project"
    if args.release:
        action += " (with host releasing)"
    _question_user_on_host_action(args, action + " for")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_check=False, with_auto_healing=False))
    return _process_host_id_args(args, lambda host_id:
        client.switch_project(host_id, args.project, release=args.release, erase_disks=args.erase_disks,
                              force=args.force, restrictions=parse_arg_list(args.restrictions), **params))


def _on_release_host(client, args):
    action = "release host"
    _question_user_on_host_action(args, action + " for")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_check=False, with_auto_healing=False))
    return _process_host_id_args(args, lambda host_id: client.release_host(host_id,
                                                                           erase_disks=args.erase_disks, **params))


def _on_power_on(client, args):
    _question_user_on_host_action(args, "power on")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_cms=False))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.power_on_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.POWER_ON, params)


def _maintenance_params(args, with_power_off=True):
    params = {}
    keys = ["ticket_key", "timeout_status", "operation_state", "cms_task_action"]
    if with_power_off:
        keys.append("power_off")

    for key in keys:
        params[key] = getattr(args, key)

    params["timeout_time"] = parse_time(args.ttl, future_period=True)

    return params


def _on_power_off(client, args):
    _question_user_on_host_action(args, "power off")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_check=False))
    params.update(_maintenance_params(args, with_power_off=False))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.power_off_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.POWER_OFF, params)


def _on_reboot(client, args):
    _question_user_on_host_action(args, "reboot")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args))
    params.update(drop_none({"ssh": args.ssh}))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.reboot_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.REBOOT, params)


def _on_check_dns(client, args):
    _question_user_on_host_action(args, "check DNS records for")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_cms=False))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.check_host_dns(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.CHECK_DNS, params)


def _on_profile(client, args):
    _question_user_on_host_action(args, "profile")

    params = _host_action_kwargs(args)
    params.update(drop_none({
        "profile": args.profile,
        "profile_tags": parse_arg_list(args.profile_tags),
        "redeploy": args.redeploy,
        "provisioner": args.provisioner,
        "config": args.config,
        "deploy_config_policy": args.deploy_config_policy,
        "deploy_tags": args.deploy_tags,
        "deploy_network": args.deploy_network,
    }))
    params.update(_task_kwargs(args))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.profile_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.PROFILE, params)


def _on_redeploy(client, args):
    _question_user_on_host_action(args, "redeploy")

    params = _host_action_kwargs(args)
    params.update(drop_none({
                                 "config": args.config,
                                 "deploy_config_policy": args.deploy_config_policy,
                                 "provisioner": args.provisioner,
                                 "deploy_tags": parse_arg_list(args.deploy_tags),
                                 "deploy_network": args.deploy_network
    }))
    params.update(_task_kwargs(args))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.redeploy_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.REDEPLOY, params)


def _on_prepare(client, args):
    _question_user_on_host_action(args, "prepare")

    params = _host_action_kwargs(args)
    params.update(drop_none({
        "skip_profile": args.skip_profile,
        "profile": args.profile,
        "profile_tags": parse_arg_list(args.profile_tags),
        "provisioner": args.provisioner,
        "config": args.config,
        "deploy_config_policy": args.deploy_config_policy,
        "deploy_tags": parse_arg_list(args.deploy_tags),
        "deploy_network": args.deploy_network,
        "restrictions": parse_arg_list(args.restrictions),
        "keep_name": args.keep_name
    }))
    params.update(_task_kwargs(args))

    if args.group is None:
        return _process_host_id_args(args, lambda host_id: client.prepare_host(host_id, **params))
    else:
        _process_group_operation(client, args, task_processing.TaskType.PREPARE, params)


def _on_remove(client, args):
    _question_user_on_host_action(args, "remove")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_check=False, with_auto_healing=False))
    remove_host = partial(client.remove_host, lui=args.lui, instant=args.instant, **params)
    return _process_host_id_args(args, remove_host)


def _on_handle_failure(client, args):
    _question_user_on_host_action(args, "handle host failure")

    params = _host_action_kwargs(args)
    params.update(drop_none({"ignore_cms": True if args.ignore_cms else None}))

    _handle = lambda host_id: print_object("result", client.handle_host_failure(host_id, **params))
    return _process_host_id_args(args, _handle)


def _on_get_deploy_log(client, args):
    for line in client.get_deploy_log(args.id, provisioner=args.provisioner, tail_bytes=args.tail_bytes):
        print(line)


def _on_get_profile_log(client, args):
    for line in client.get_profile_log(args.id):
        print(line)


def _on_clear_deploy_config(client, args):
    params = _host_action_kwargs(args)
    return _process_host_id_args(args, lambda host_id: client.clear_deploy_config(host_id, **params))


def _on_set_deploy_config(client, args):
    params = _host_action_kwargs(args)
    return _process_host_id_args(args, lambda host_id: client.set_deploy_config(
        host_id,
        config_name=args.config,
        provisioner=args.provisioner,
        deploy_config_policy=args.deploy_config_policy,
        deploy_tags=parse_arg_list(args.deploy_tags),
        deploy_network=args.deploy_network,
        **params
    ))


def _question_user_on_host_action(args, action):
    if args.group is not None and args.all:
        question_user(args, "Are you sure want to {} all hosts of '{}' group?", action, args.group)
    elif args.id:
        question_user(args, "Are you sure want to {} hosts identified as: {}?", action, ", ".join(args.id))


def _get_args(args):
    if args:
        return args

    if sys.stdin.isatty():
        print("Awaiting a list of hosts from stdin...")

    args = []

    for line in sys.stdin:
        comment_pos = line.find("#")
        if comment_pos != -1:
            line = line[:comment_pos]

        line = line.strip()
        if line:
            args += re.split(r"\s+", line)

    return args


def _get_host_args(args, hosts, split=False, with_host_groups=True):
    # Notice: The function mustn't reorder hosts: we should process them in exact order that they are specified.

    invs, names = [], []

    if with_host_groups and args.group is not None and args.all:
        if hosts:
            raise Error("No hosts must be specified when --all option is used.")

        names = _get_group_hosts(args.group)
    else:
        inv_regex = re.compile(r"^\d+$")

        for host_id in _get_args(hosts):
            if inv_regex.search(host_id):
                invs.append(int(host_id))
            else:
                names.append(host_id)

        if with_host_groups and args.group is not None and invs:
            raise Error("Inventory numbers aren't supported yet for host groups.")

        if invs and names:
            raise Error("You must specify either host inventory numbers or host FQDNs - not both")

        if names:
            if args.syntax == _SYNTAX_PLAIN:
                _validate_fqdns(names)
            elif args.syntax == _SYNTAX_BLINOV:
                # Try to not load heavy modules when it's not needed to make CLI more responsive
                from library.sky.hostresolver.resolver import Resolver
                expressions = names[:]
                names = []
                for expr in expressions:
                    names.extend(Resolver().resolveHosts(expr))
            else:
                raise LogicalError()

        if with_host_groups and names and args.group is not None:
            invalid = set(names) - set(_get_group_hosts(args.group))
            if invalid:
                raise Error("The following hosts doesn't belong to host group '{}': {}.",
                            args.group, ", ".join(invalid))

    invs, names = unique_list(invs), unique_list(names)

    return (invs, names) if split else invs + names


def _process_host_id_args(args, func):
    return process_args(_get_host_args(args, args.id), func)


def _get_group_hosts(group_name):
    return host_groups.get_group(group_name)["hosts"]


def _process_group_operation(client, args, task_type, params=None):
    group = host_groups.get_group(args.group, locked=True)

    try:
        if "task" in group:
            raise Error("Unable to schedule '{}' task for group '{}': it's already processing '{}' task.",
                        task_type, group["name"], group["task"]["type"])

        hosts = _get_host_args(args, args.id)
        group["task"] = {
            "type": task_type,
            "hosts": hosts,
            "params": params,
        }
        group.save()

        task_processing.process_task(client, group)
    finally:
        group.unlock()


def _validate_fqdns(names):
    """Simple FQDN validation to get errors for `wl h lo host-list` instead of `wl h lo < host-list`."""

    fqdn_re = re.compile(r"^([a-z0-9]+(-[a-z0-9]+)*\.)+[a-z]{2,}$", re.IGNORECASE)

    for name in names:
        if not fqdn_re.search(name):
            raise Error("Invalid FQDN: {}.", name)


def _filter_dict(d, keys):
    return {key: value for key, value in d.items() if key in keys}


def _display_hosts(hosts, columns, batch, total=None):
    for host in hosts:
        if "macs" in host:
            host["macs"] = ",".join(host["macs"])

        if "task" in host:
            task = host["task"]
            for field in ("owner", "status", "status_message", "error"):
                if field in task:
                    host["task_" + field] = task[field]
            if "audit_log_id" in task:
                host["task_log_id"] = task["audit_log_id"]

        if "custom_deploy_configuration" in host:
            if host["custom_deploy_configuration"]:
                for field in ("provisioner", "config"):
                    if field in host:
                        host[field] = "@" + host[field]

                if "deploy_tags" in host:
                    host["deploy_tags"] = "@" + ",".join(host["deploy_tags"])

                host["custom_deploy_configuration"] = "yes"
            else:
                if "deploy_tags" in host:
                    host["deploy_tags"] = ",".join(host["deploy_tags"])
                host["custom_deploy_configuration"] = "no"

        if "restrictions" in host:
            host["restrictions"] = ",".join(host["restrictions"])

        if "health" in columns and "health" in host:
            if host["health"]["status"] == "ok":
                host["health"] = "ok"
            else:
                reasons = [
                    reason[:-len("-failed")] if reason.endswith("-failed") else reason
                    for reason in host["health"]["reasons"]
                ]

                host["health"] = ",".join(reasons)

        if "location" in host:
            location = host["location"]

            if "switch" in location:
                host["switch"] = location["switch"]

            if "port" in location:
                host["port"] = location["port"]

            if "network_source" in location:
                host["netmap_source"] = location["network_source"]
            if "geo_location" in columns:
                host["geo_location"] = _display_geo_location(location)
            if "geo_time" in columns and "physical_timestamp" in location:
                host["geo_time"] = format_time(location["physical_timestamp"])

        if "active_mac_time" in host:
            host["active_mac_time"] = format_time(host["active_mac_time"])

        if "ips_time" in host:
            host["ips_time"] = format_time(host["ips_time"])

        if "network_timestamp" in host:
            host["netmap_time"] = format_time(host["network_timestamp"])

        if "messages" in host:
            messages_list = []
            for module, messages in sorted(host["messages"].items()):
                for message in sorted(messages, key=itemgetter("severity")):
                    messages_list.append("{module}: {severity}: {message}".format(
                        module=module, severity=message["severity"], message=message["message"]
                    ))
            host["messages"] = "\n".join(messages_list)
    _TABLE_VIEW.render("hosts", hosts, columns, batch, total=total)


def _init_fqdn_deinvalidation_operation(subparsers):
    parser = add_host_action(subparsers, "fqdn-deinvalidation", _on_fqdn_deinvalidation,
                             "Sync FQDN between Bot and Wall-e", short="fqdn-d", with_reason=True)
    parser.add_argument("--release", default=False, action="store_true", help="release host after sync")
    parser.add_argument("--clear-old-fqdn-records", default=False, action="store_true",
                        help="Clear old DNS records, remove old fqdn from LUI")
    _add_new_task_arguments(parser, with_check=False, with_auto_healing=False)


def _on_fqdn_deinvalidation(client, args):
    _question_user_on_host_action(args, "fqdn-deinvalidation")

    params = _host_action_kwargs(args)
    params.update(_task_kwargs(args, with_check=False, with_auto_healing=False))

    fqdn_deinvalidation = partial(client.fqdn_deinvalidation,
                                  release=args.release, clear_old_fqdn_records=args.clear_old_fqdn_records, **params)
    return _process_host_id_args(args, fqdn_deinvalidation)
