"""Wall-E CLI main module."""

import argparse
import os
import sys

import walle_cli.setup
from walle_api.client import WalleClient
from walle_api.constants import version as sdk_version, PRODUCTION_URL, TESTING_URL

import argcomplete
import walle_cli.access_token
import walle_cli.audit_log
import walle_cli.dc_shortnames
import walle_cli.deploy_configs
import walle_cli.health_checks
import walle_cli.hosts
import walle_cli.host_groups
import walle_cli.maintenance
import walle_cli.preorders
import walle_cli.projects
import walle_cli.scenario
import walle_cli.settings
import walle_cli.operation_log
import walle_cli.automation_plots
from walle_cli import constants
from walle_cli.common import NegativeUserAnswerError, handle_error, set_short_mode, parser_help


def entry_point():
    binary = os.path.basename(sys.argv[0])

    if binary == "wl":
        return main(short=True)
    else:
        return main(short=False)


def main(short=False):
    set_short_mode(short=short)
    args = _parse_args(short=short)

    try:
        if args.wall_e is not None:
            url = args.wall_e
        elif args.testing:
            url = TESTING_URL
        else:
            url = PRODUCTION_URL

        token = args.access_token
        if token is None:
            token = walle_cli.access_token.get_token()

        client = WalleClient(name="Wall-E.CLI/" + constants.version, url=url, access_token=token)
        result = args.handle(client, args)
    except NegativeUserAnswerError:
        print("Abort the operation.", file=sys.stderr)
        result = False
    except Exception as e:
        handle_error(e)
        result = False

    sys.exit(os.EX_SOFTWARE if result is False else os.EX_OK)


def _parse_args(short):
    # Parse the version argument manually because we have non-optional arguments
    if "-V" in sys.argv or "--version" in sys.argv:
        print("Wall-E Cli: {}".format(constants.version))
        print("Wall-E SDK: {}".format(sdk_version))
        print("Python: {}".format(sys.version))
        sys.exit(os.EX_OK)

    description = [
        "Command-line interface to Wall-E",
        "",
        "To enable tab-completion (works in bash and zsh), add the following to your shell login script:",
        "  eval \"$(register-python-argcomplete wall-e)\"",
        "  eval \"$(register-python-argcomplete wl)\"",
    ]
    if short:
        description[0] += " (short version)"
    description = "\n".join(description)

    parser = argparse.ArgumentParser(description=description, formatter_class=argparse.RawDescriptionHelpFormatter)
    parser.set_defaults(handle=parser_help(parser))

    stand_group = parser.add_mutually_exclusive_group()
    stand_group.add_argument("-u", "--wall-e", metavar="HOST", help="go to the specified Wall-E installation")
    stand_group.add_argument("-P", "--production", action="store_true", help="go to production Wall-E installation")
    stand_group.add_argument("-T", "--testing", action="store_true", help="go to testing Wall-E installation")

    parser.add_argument("-t", "--access-token", help="access token to use instead of obtained one")

    output = parser.add_mutually_exclusive_group()
    output.add_argument("-b", "--batch", action="store_true",
                        help="output data in script friendly form (no indentation or headers)")
    parser.add_argument("-j", "--json", action="store_true", help="output data in JSON")

    parser.add_argument("-y", "--yes", action="store_true",
                        help="assume 'yes' as answer to all prompts and run non-interactively")
    parser.add_argument("-V", "--version", action="store_true", help="print the version and exit")

    subparsers = parser.add_subparsers(required=True, dest="subcommand")
    walle_cli.projects.init(subparsers)
    walle_cli.preorders.init(subparsers)
    walle_cli.hosts.init(subparsers)
    walle_cli.host_groups.init(subparsers)
    walle_cli.deploy_configs.init(subparsers)
    walle_cli.audit_log.init(subparsers)
    walle_cli.settings.init(subparsers)
    walle_cli.access_token.init(subparsers)
    walle_cli.scenario.init(subparsers)
    walle_cli.maintenance.init(subparsers)
    walle_cli.health_checks.init(subparsers)
    walle_cli.dc_shortnames.init(subparsers)
    walle_cli.operation_log.init(subparsers)
    walle_cli.automation_plots.init(subparsers)

    argcomplete.autocomplete(parser)

    return parser.parse_args()
