"""Maintenance works management."""

from walle_cli.common import (
    TableView, Column, register_subparsers, register_parser, question_user, format_time)


class MaintenanceStatus:
    CREATED = "created"
    STARTED = "started"
    READY = "ready"
    FINISHING = "finishing"
    FINISHED = "finished"
    CANCELING = "canceling"
    CANCELED = "canceled"
    REJECTED = "rejected"
    ALL = [CREATED, STARTED, READY, CANCELING, CANCELED, REJECTED, FINISHING, FINISHED]
    ALL_TERMINATION = [REJECTED, CANCELING, CANCELED, FINISHING, FINISHED]
    ALL_CURRENT = [CREATED, STARTED, READY, CANCELING, REJECTED, FINISHING]


_MAINTENANCE_TABLE_VIEW = TableView(
    default_columns=(
        Column("id", "ID"),
        Column("status", "Status"),
        Column("ticket_key", "Ticket"),
        Column("action_time", "Last update"),
        Column("name", "Name"),
    ),
    other_columns=(
        Column("issuer", "Issuer"),
        Column("hosts", "Number of hosts"),
    )
)


_HOSTS_TABLE_VIEW = TableView(
    default_columns=(
        Column("inv", "Inventory number"),
        Column("status", "Status"),
        Column("state", "State")
    )
)


def init(subparsers):
    subparsers = register_subparsers(subparsers, "maintenance", "Maintenance scenario management", short="m")
    _init_list_maintenance(subparsers)
    _init_add_maintenance(subparsers)

    _add_maintenance_status_handler(subparsers, "start", _on_maintenance_start, "Start maintenance")
    _add_maintenance_status_handler(subparsers, "finish", _on_maintenance_finish, "Finish maintenance")
    _add_maintenance_status_handler(subparsers, "cancel", _on_maintenance_cancel, "Cancel maintenance")


def _init_list_maintenance(subparsers):
    parser = register_parser(subparsers, "list", _on_list, "Show all scenarios")
    parser.add_argument("--id", help="filter by scenario id")
    parser.add_argument("-n", "--name", help="filter by name")
    parser.add_argument("-t", "--ticket-key", help="filter by ticket key")
    parser.add_argument("-i", "--issuer", help="filter by issuer")
    parser.add_argument("-s", "--status", default=MaintenanceStatus.ALL_CURRENT, help="filter by scenario status")
    parser.add_argument("-l", "--limit", type=int, default=100,
                        help="maximum number of returned hosts (default is 100)")
    # parser.add_argument("--columns", help="a comma-separated list of columns to output ({})".format(
    #     ", ".join(_MAINTENANCE_TABLE_VIEW.get_column_ids())))


def _on_list(client, args):
    params = {
        "scenario_id": args.id,
        "name":  args.name,
        "issuer": args.issuer,
        "ticket_key": args.ticket_key,
        "status": args.status,
        "limit": args.limit,
    }
    columns, fields = _MAINTENANCE_TABLE_VIEW.parse_column_list(column_ids_string=None)
    works = client.list_noc_maintenance(**params)
    _display_scenarios(works["result"], columns, args.batch, works["total"])


def _display_scenarios(works, columns, batch, total):
    for maintenance in works:
        if "action_time" in maintenance:
            maintenance["action_time"] = format_time(maintenance["action_time"])
    _MAINTENANCE_TABLE_VIEW.render("maintenance", works, columns, batch, total=total)


def _init_add_maintenance(subparsers):
    parser = register_parser(subparsers, "add", _on_add_maintenance, help="Add maintenance", with_reason=True)

    parser.add_argument("-s", "--switch", help="The switch to collect hosts from")
    parser.add_argument("-n", "--name", help="Name of the scenario")
    parser.add_argument("-t", "--ticket-key", help="Key of related ticket")


def _on_add_maintenance(client, args):
    client.create_noc_maintenance(switch=args.switch, name=args.name, ticket_key=args.ticket_key, reason=args.reason)


def _add_maintenance_status_handler(subparsers, command, handler, help, short=None, with_reason=True):
    parser = register_parser(subparsers, command, handler, help,
                             short=short, with_reason=with_reason)
    parser.add_argument("scenario_id", help="Maintenance Scenario id")
    return parser


def _on_maintenance_start(client, args):
    maintenance = client.get_noc_maintenance_status(args.scenario_id)
    question_user(args, "Are you sure you want to start maintenance '{}' #{}?", maintenance["name"], args.scenario_id)

    client.start_noc_maintenance(scenario_id=args.scenario_id, reason=args.reason)


def _on_maintenance_finish(client, args):
    maintenance = client.get_noc_maintenance_status(args.scenario_id)
    question_user(args, "Are you sure you want to finish maintenance '{}' #{} in '{}' status?",
                  maintenance["name"], args.scenario_id, maintenance["status"])

    client.cfinish_noc_maintenance(scenario_id=args.scenario_id, reason=args.reason)


def _on_maintenance_cancel(client, args):
    maintenance = client.get_noc_maintenance_status(args.scenario_id)
    question_user(args, "Are you sure you want to cancel maintenance '{}' #{} in '{}' status?",
                  maintenance["name"], args.scenario_id, maintenance["status"])

    client.cancel_noc_maintenance(scenario_id=args.scenario_id, reason=args.reason)
