"""Simple data model on file system."""

import errno
import fcntl
import os
import re

import object_validator
import yaml
from walle_cli.common import get_config_path, Error


class ObjectDoesntExistError(Error):
    def __init__(self):
        super(ObjectDoesntExistError, self).__init__("Object doesn't exist.")


class ObjectAlreadyExistsError(Error):
    def __init__(self):
        super(ObjectAlreadyExistsError, self).__init__("The object already exists.")


class ObjectValidationError(Error):
    def __init__(self, error):
        super(ObjectValidationError, self).__init__("Object validation error: {}", error)


class Model(dict):
    id_regex = re.compile(r"^[a-zA-Z0-9_-]+$")
    """All object IDs must match this regex."""

    def __init__(self, data):
        super(dict, self).__init__()
        self.update(data)

        self._lock = None
        self.__last_saved_data = None

    @classmethod
    def get_object(cls, obj_id, locked=False):
        cls.__validate_id(obj_id)

        path = cls.__get_obj_path(obj_id)
        if not os.path.exists(path):
            raise ObjectDoesntExistError()

        lock = cls.__lock(obj_id) if locked else None

        try:
            try:
                with open(path, "rb") as obj_file:
                    obj = yaml.safe_load(obj_file)
            except Exception as e:
                if isinstance(e, EnvironmentError) and e.errno == errno.ENOENT:
                    raise ObjectDoesntExistError()

                raise Error("Error while reading '{}': {}.", path, e)

            obj[cls._id_field] = obj_id
            obj = cls(obj)

            try:
                obj.validate()
            except ObjectValidationError as e:
                raise Error("Error while reading '{}': {}", path, e)

            if lock is not None:
                obj._lock = lock

            return obj
        except:
            cls.__unlock(lock)
            raise

    @classmethod
    def list_objects(cls):
        path = cls.__get_collection_path()

        try:
            file_names = os.listdir(path)
        except EnvironmentError as e:
            if e.errno == errno.ENOENT:
                file_names = []
            else:
                raise Error("Failed to read contents of '{}' directory: {}.", path, e.strerror)

        obj_ids = []
        for file_name in file_names:
            if file_name.endswith(".yaml") and not file_name.startswith("."):
                obj_id = file_name[:-5]
                if cls.id_regex.search(obj_id):
                    obj_ids.append(obj_id)

        return obj_ids

    @property
    def id(self):
        return self[self._id_field]

    def lock(self):
        if self._lock is not None:
            return False

        self.__validate_id(self.id)
        self._lock = self.__lock(self.id)
        return True

    def unlock(self):
        if self._lock is not None:
            self.__unlock(self._lock)
            self._lock = None

    def remove(self):
        self.__validate_id(self.id)

        removed = False
        own_lock = self.lock()

        try:
            for path in self.__get_tmp_obj_path(self.id), self.__get_obj_path(self.id):
                try:
                    os.unlink(path)
                except EnvironmentError as e:
                    if e.errno != errno.ENOENT:
                        raise Error("Unable to delete file '{}': {}.", path, e.strerror)

            removed = True
        finally:
            if removed or own_lock:
                self.unlock()

    def save(self, force_creation=False, only_if_changed=False):
        self.validate()
        data = self.to_yaml()

        if only_if_changed and data == self.__last_saved_data:
            return

        self.__get_collection_path(ensure_exists=True)

        own_lock = self.lock()
        try:
            path = self.__get_obj_path(self.id)
            if force_creation and os.path.exists(path):
                raise ObjectAlreadyExistsError()

            tmp_path = self.__get_tmp_obj_path(self.id)

            try:
                with open(tmp_path, "w") as tmp_obj:
                    tmp_obj.write(data)
            except EnvironmentError as e:
                raise Error("Unable to save '{}': {}.", tmp_path, e.strerror)

            try:
                os.rename(tmp_path, path)
            except EnvironmentError as e:
                raise Error("Unable to rename '{}' to '{}': {}", tmp_path, path, e.strerror)
        finally:
            if own_lock:
                self.unlock()

        self.__last_saved_data = data

    def get_path(self):
        return self.__get_obj_path(self.id)

    def to_yaml(self):
        obj = self.copy()
        del obj[self._id_field]
        return yaml.safe_dump(obj)

    def validate(self):
        self.__validate_id(self.id)

        try:
            object_validator.validate("obj", self.copy(), self._scheme)
        except object_validator.ValidationError as e:
            raise ObjectValidationError(e)

    @classmethod
    def __lock(cls, obj_id):
        lock = None
        obj_path = cls.__get_obj_path(obj_id)
        lock_path = cls.__get_lock_path(obj_id)

        try:
            lock = open(lock_path, "wb")

            try:
                fcntl.flock(lock, fcntl.LOCK_EX | fcntl.LOCK_NB)
            except EnvironmentError as e:
                if e.errno == errno.EWOULDBLOCK:
                    raise Error("Unable to lock '{}': it's already locked.", obj_path)
                raise
        except Exception as e:
            if lock is not None:
                lock.close()

            if isinstance(e, EnvironmentError):
                raise Error("Unable to flock '{}': {}.", lock_path, e.strerror)

            raise

        return lock

    @staticmethod
    def __unlock(lock):
        if lock is None:
            return

        try:
            os.unlink(lock.name)
        except EnvironmentError:
            pass

        lock.close()

    @classmethod
    def __get_obj_path(cls, obj_id):
        return os.path.join(cls.__get_collection_path(), obj_id + ".yaml")

    @classmethod
    def __get_tmp_obj_path(cls, obj_id):
        return cls.__get_obj_path(obj_id) + ".tmp"

    @classmethod
    def __get_lock_path(cls, obj_id):
        return cls.__get_obj_path(obj_id) + ".lock"

    @classmethod
    def __get_collection_path(cls, ensure_exists=False):
        config_path = get_config_path(ensure_exists=ensure_exists)
        collection_path = os.path.join(config_path, cls._collection_name)

        if ensure_exists:
            try:
                os.mkdir(collection_path)
            except EnvironmentError as e:
                if e.errno != errno.EEXIST:
                    raise Error("Unable to create '{}' directory: {}.", collection_path, e.strerror)

        return collection_path

    @classmethod
    def __validate_id(cls, obj_id):
        if not cls.id_regex.search(obj_id):
            raise ObjectValidationError("Invalid object ID: {}".format(obj_id))
