"""Operation log management."""

import sys
from collections import deque

try:
    import json
except ImportError:
    import simplejson as json

from walle_cli.common import register_subparsers, register_parser, get_supported_time_formats, parse_time, format_time


def init(subparsers):
    subparsers = register_subparsers(subparsers, "operation-log", "Operation log management", short="oplog")

    parser = register_parser(subparsers, "get", _on_get, "Get the log")

    parser.add_argument("-i", "--id", help="filter by ids separated by commas")
    parser.add_argument("-a", "--audit_log_id", help="filter by audit log ids separated by commas")
    parser.add_argument("-I", "--inv", help="filter by invs separated by commas")
    parser.add_argument("-n", "--name", help="filter by fqdn's separated by commas")
    parser.add_argument("-S", "--scenario_id", help="filter by scenario ID's separated by commas")
    parser.add_argument("-t", "--type", help="filter by oplog types separated by commas")

    supported_formats = get_supported_time_formats()
    parser.add_argument("-s", "--start-time", help="time lower bound in the following format: " + supported_formats)
    parser.add_argument("-e", "--end-time", help="time upper bound in the following format: " + supported_formats)


def _on_get(client, args):
    start_time = parse_time(args.start_time, future_period=False)
    end_time = parse_time(args.end_time, future_period=False)

    fields = ["time", "id", "audit_log_id", "inv", "name", "type", "scenario_id"]

    if start_time is None and end_time is None:
        reverse = True
        limit = 100
    else:
        reverse = False
        limit = None

    entries = client.iter_operation_log(
        id=args.id, audit_log_id=args.audit_log_id, inv=args.inv, name=args.name,
        scenario_id=args.scenario_id, type=args.type,
        start_time=start_time, end_time=end_time, limit=limit, fields=fields, reverse=reverse
    )

    if reverse:
        entries = list(entries)
        entries.reverse()

    entry_page = deque()

    for entry in entries:
        entry_page.append(entry)

        if len(entry_page) >= client.MAX_PAGE_SIZE:
            _display_page(entry_page)
            entry_page.clear()

    if entry_page:
        _display_page(entry_page)

    if limit is not None and not args.batch:
        print("> Listed last {limit} entries. "
              "Use --start-time and --end-time to list all entries for the specified period.".format(limit=limit), file=sys.stderr)


def _display_page(entries):
    id_len = max(len(entry["id"]) for entry in entries)
    type_len = max(len(entry["type"]) for entry in entries)

    for entry in entries:
        log_line = ("{time} [{id:>"+str(id_len)+"}] [{type:>"+str(type_len)+"}]").format(
            time=format_time(entry["time"]), id=entry["id"], type=entry["type"])

        audit_log_id = " [#{}] ".format(entry["audit_log_id"]) if "audit_log_id" in entry else ""

        log_line += audit_log_id

        inv_part = "#{}".format(entry["inv"]) if "inv" in entry else None

        host_id_section = " [{}]".format(
            "|".join(str(part) for part in (inv_part, entry.get("name"), entry.get("scenario_id")) if part is not None)
        )
        log_line += host_id_section

        print(log_line)
