"""Preorder management."""

from walle_api.constants import Provisioner, Restriction

from walle_cli.common import TableView, Column, register_subparsers, register_parser, parse_arg_list, render_result, \
    question_user, format_choices

_TABLE_VIEW = TableView(
    default_columns=(
        Column("id", "ID"),
        Column("owner", "Owner"),
        Column("issuer", "Issuer"),
        Column("project", "Project", align=Column.ALIGN_CENTER),
        Column("prepare", "Prepare", align=Column.ALIGN_CENTER),
        Column("processed", "Processed", align=Column.ALIGN_CENTER),
    ),
    other_columns=(
        Column("provisioner", "Provisioner", align=Column.ALIGN_CENTER),
        Column("deploy_config", "Deploy config", align=Column.ALIGN_CENTER),
        Column("restrictions", "Restrictions", align=Column.ALIGN_CENTER),
    )
)


def init(subparsers):
    subparsers = register_subparsers(subparsers, "preorders", "Preorder management", short="o")

    parser = register_parser(subparsers, "list", _on_list, "List all preorders registered in Wall-E")
    parser.add_argument("-C", "--columns", help="a comma-separated list of columns to output ({})".format(
                        ", ".join(_TABLE_VIEW.get_column_ids())))

    _add_preorder_action(subparsers, "get", _on_get, "Show all information about the specified preorder")

    parser = _add_preorder_action(subparsers, "add", _on_add, "Add preorder to Wall-E", with_sudo=True, with_reason=True)
    parser.add_argument("project", help="project ID to add the hosts to")
    parser.add_argument("--prepare", action="store_true", default=False, help="prepare taken hosts")
    parser.add_argument("-p", "--provisioner", choices=format_choices(Provisioner.ALL), help="provisioner")
    parser.add_argument("-c", "--config", help="config to deploy the hosts with")
    parser.add_argument("-r", "--restrictions",
                        help="a comma-separated list of restrictions to set for the hosts ({})".format(
                            ", ".join(Restriction.ALL)))

    _add_preorder_action(subparsers, "restart", _on_restart, short="s", description="Restart preorder processing",
                         with_sudo=True, with_reason=True)

    _add_preorder_action(subparsers, "remove", _on_remove, description="Remove preorder from Wall-E",
                         with_sudo=True, with_reason=True)


def _add_preorder_action(subparsers, name, handler, description, short=None, with_sudo=False, with_reason=False):
    parser = register_parser(subparsers, name, handler, description,
                             short=short, with_sudo=with_sudo, with_reason=with_reason)
    parser.add_argument("id", type=int, help="preorder ID")
    return parser


def _on_list(client, args):
    columns, fields = _TABLE_VIEW.parse_column_list(args.columns)
    preorders = client.get_preorders(fields=fields)["result"]

    for preorder in preorders:
        for key, value in preorder.items():
            if isinstance(value, bool):
                preorder[key] = "yes" if value else "no"

        if "restrictions" in preorder:
            preorder["restrictions"] = ",".join(preorder["restrictions"])

    _TABLE_VIEW.render("preorders", preorders, columns, args.batch)


@render_result
def _on_get(client, args):
    preorder = client.get_preorder(args.id, fields=_TABLE_VIEW.get_column_fields() + [
        "acquired_hosts", "failed_hosts", "messages", "errors"])
    preorder.pop("id", None)
    return {"preorder": preorder}


def _on_add(client, args):
    question_user(args, "Are you sure want to add #{} preorder to Wall-E?", args.id)
    client.add_preorder(args.id, project=args.project, prepare=args.prepare, provisioner=args.provisioner,
                        deploy_config=args.config, restrictions=parse_arg_list(args.restrictions),
                        sudo=args.sudo, reason=args.reason)


def _on_restart(client, args):
    question_user(args, "Are you sure want to restart #{} preorder processing?", args.id)
    client.restart_preorder(args.id, sudo=args.sudo, reason=args.reason)


def _on_remove(client, args):
    question_user(args, "Are you sure want to remove #{} preorder from Wall-E?", args.id)
    client.remove_preorder(args.id, sudo=args.sudo, reason=args.reason)
