"""Project management."""

import re

from walle_api.client import WalleClient
from walle_api.constants import Restriction, AutomationType, CmsApiVersion, DeployNetwork, VlanScheme, MtnIpMethod

from walle_cli.args_utils import ActionStoreCredit, ActionAppendLimit, StoreKeyValuePair
from walle_cli.common import TableView, Column, Error, filter_none, register_subparsers, \
    register_parser, parse_arg_list, print_object, question_user, period_format, parse_period, parse_limit, format_time, \
    get_supported_time_formats, parse_time, split_comma_separated_values_in_dict, render_result

_TABLE_VIEW = TableView(
    default_columns=(
        Column("id", "ID"),
        Column("name", "Name", align=Column.ALIGN_LEFT),
        Column("automation", "Automation", fields=("healing_automation.enabled", "dns_automation.enabled"),
               align=Column.ALIGN_CENTER),
        Column("fsm_handbrake", "FSM handbrake",
               fields=("fsm_handbrake.timeout_time", "fsm_handbrake.issuer",
                       "fsm_handbrake.ticket_key", "fsm_handbrake.reason"),
               align=Column.ALIGN_CENTER),
        Column("default_host_restrictions", "Default host restrictions", align=Column.ALIGN_CENTER),
    ),
    other_columns=(
        Column("tags", "Tags", align=Column.ALIGN_LEFT),
        Column("owners", "Owners", align=Column.ALIGN_CENTER),
        Column("vlan_scheme", "VLAN scheme", align=Column.ALIGN_CENTER),
        Column("native_vlan", "Native VLAN", align=Column.ALIGN_RIGHT),
        Column("extra_vlans", "Extra VLANs", align=Column.ALIGN_CENTER),
        Column("dns_domain", "DNS domain", align=Column.ALIGN_CENTER),
        Column("host_shortname_template", "Host ShortName template", align=Column.ALIGN_CENTER),
        Column("hbf_project_id", "HBF project id", align=Column.ALIGN_CENTER),
        Column("bot_project_id", "BOT/OEBS project id", align=Column.ALIGN_CENTER),

        Column("profile", "Profile", align=Column.ALIGN_RIGHT),
        Column("profile_tags", "Profile tags", align=Column.ALIGN_CENTER),

        Column("provisioner", "Provisioner", align=Column.ALIGN_CENTER),
        Column("deploy_config", "Deploy config", align=Column.ALIGN_RIGHT),
        Column("deploy_config_policy", "Deploy config policy", align=Column.ALIGN_CENTER),
        Column("deploy_tags", "Deploy tags", align=Column.ALIGN_CENTER),
        Column("certificate_deploy", "Deploy certificate", align=Column.ALIGN_RIGHT),
        Column("reboot_via_ssh", "Reboot via SSH", align=Column.ALIGN_RIGHT),
        Column("roles", "Project roles", align=Column.ALIGN_RIGHT),
        Column("repair_request_severity", "Request severity", align=Column.ALIGN_RIGHT),
        Column("cms_settings", "Cms settings", align=Column.ALIGN_RIGHT),
    )
)

_NOTIFICATION_SEVERITIES = ("audit", "info", "warning", "bot", "error", "critical")


def init(subparsers):
    subparsers = register_subparsers(subparsers, "projects", "Project management")

    parser = register_parser(subparsers, "list", _on_list, "List all registered projects")
    parser.add_argument("-C", "--columns", help="a comma-separated list of columns to output ({})".format(
                        ", ".join(_TABLE_VIEW.get_column_ids())))
    parser.add_argument("-t", "--tags", help="filter by projects' tags")

    _add_project_action(subparsers, "get", _on_get, "Show all information about the specified project")

    parser = _add_project_action(subparsers, "add", _on_add, "Add project to the system", with_reason=True)
    parser.add_argument("name", help="name that will identify the project in UI")
    parser.add_argument("tags", nargs="?", help="comma-separated list of tags to assign to project")
    parser.add_argument("-o", "--owners", metavar="LOGINS", help="a comma-separated list of project owners")
    parser.add_argument("-P", "--provisioner", help="default provisioner name to use", required=True)
    parser.add_argument("-d", "--deploy-config", help="default provisioner config name to use", required=True)
    parser.add_argument("-t", "--deploy-tags", help="a comma-separated list of default Einstellung tags")
    parser.add_argument("-n", "--deploy-network", choices=DeployNetwork.ALL,
                        help="deploy host in project or service network")
    parser.add_argument("--dns-domain", help="domain for DNS auto configuration")
    parser.add_argument("--host-shortname-template", help="Custom template for host shortname")
    parser.add_argument("--hbf-project-id", help="HBF project id for HBF/MTN projects")
    parser.add_argument("--ip-method", choices=MtnIpMethod.ALL,
                        help="Method for generating host-id part of IP-address for HBF/MTN")
    parser.add_argument("--bot-project-id", help="BOT/OEBS project id")
    parser.add_argument("--certificate-deploy", help="Enable deployment of certificates",
                        action="store_true", dest="certificate_deploy", default=None)
    parser.add_argument("--reboot-via-ssh", help="Enable rebooting via SSH",
                        action="store_true", dest="reboot_via_ssh", default=None)

    _add_restrictions_argument(parser)

    parser = _add_project_action(subparsers, "modify", _on_modify, "Modify project", with_reason=True)
    parser.add_argument("-n", "--name", help="Name that will identify the project in UI")
    parser.add_argument("-t", "--tags", help="project's tags")
    parser.add_argument("--bot-project-id", help="BOT/OEBS project id")
    parser.add_argument("--hbf-project-id", help="HBF project id for HBF-enabled projects")
    parser.add_argument("--ip-method", choices=MtnIpMethod.ALL,
                        help="Method for generating host-id part of IP-address for HBF/MTN")
    parser.add_argument("--host-shortname-template", help="Custom template for host shortname")
    cert = parser.add_mutually_exclusive_group()
    cert.add_argument(
        "--enable-certificate-deploy", help="Enable deployment of certificates",
        action="store_true", dest="certificate_deploy", default=None)
    cert.add_argument(
        "--disable-certificate-deploy", help="Disable deployment of certificates",
        action="store_false", dest="certificate_deploy", default=None)

    _add_restrictions_argument(parser)

    parser = _add_project_action(subparsers, "clone", _on_clone, "Clone project", with_reason=True)
    parser.add_argument("clone_id", help="clone project ID")
    parser.add_argument("clone_name", help="name of clone project that will identify it in UI")

    _init_owners(subparsers)
    _init_tags(subparsers)
    _init_owned_vlans(subparsers)
    _init_vlan_scheme(subparsers)
    _init_failure_reports(subparsers)
    _init_dns_domain(subparsers)
    _init_host_shortname_template(subparsers)
    _init_hbf_project_id(subparsers)
    _init_bot_project_id(subparsers)
    _init_host_profiling_config(subparsers)
    _init_host_provisioner_config(subparsers)
    _init_automation(subparsers)
    _init_fsm_handbrake(subparsers)
    _init_reboot_via_ssh(subparsers)
    _init_automation_plot(subparsers)
    _init_notifications(subparsers)
    _init_roles(subparsers)
    _init_repair_request_severity(subparsers)
    _init_modify_project_cms(subparsers)

    _add_project_action(subparsers, "remove", _on_remove, "Remove project from the system", with_reason=True)


def _init_owners(subparsers):
    subparsers = register_subparsers(subparsers, "owners", "Configure owners")
    _add_project_action(subparsers, "get", _on_get_owners, "Show project owners")

    for action in WalleClient.get_api_actions():
        parser = _add_project_action(subparsers, action, _get_on_modify_owners_handler(action),
                                     "{} project owners".format(action.capitalize()), with_reason=True)
        parser.add_argument("logins", help="a comma-separated list of logins to {}".format(action))


def _init_roles(subparsers):
    subparsers = register_subparsers(subparsers, "roles", "Project roles membership management", short="rl")

    parser = _add_project_action(subparsers, "get", _on_get_role_members, "Show role members")
    parser.add_argument("role", help="Project role name")

    for action, handler in [("add", _on_request_add_role_member),
                            ("remove", _on_request_remove_role_member)]:
        parser = _add_project_action(subparsers, action, handler, "Request {} role member".format(action),
                                     with_reason=True)
        parser.add_argument("role", help="Project role name")
        parser.add_argument("--member", default=None, help="Member (user or @group)")


def _init_owned_vlans(subparsers):
    subparsers = register_subparsers(subparsers, "owned-vlans", "Configure owned VLANs", short="ov")
    _add_project_action(subparsers, "get", _on_get_owned_vlans, "Show VLANs owned by project")

    for action in WalleClient.get_api_actions():
        parser = _add_project_action(subparsers, action, _get_on_modify_owned_vlans_handler(action),
                                     "{} VLANs owned by project".format(action.capitalize()), with_reason=True)
        parser.add_argument("vlans", help="a comma-separated list of VLANs to {}".format(action))


def _init_tags(subparsers):
    subparsers = register_subparsers(subparsers, "tags", "Configure project's tags", short="t")

    for action in WalleClient.get_api_actions():
        parser = _add_project_action(subparsers, action, _get_on_modify_tags_handler(action),
                                     "{} project's tags".format(action.capitalize()), with_reason=True)
        parser.add_argument("tags", nargs="?" if action == "remove" else None,
                            help="a comma-separated list of tags to {}".format(action))


def _init_vlan_scheme(subparsers):
    subparsers = register_subparsers(subparsers, "vlan-scheme", "Configure VLAN assigning scheme", short="v")
    _add_project_action(subparsers, "get", _on_get_vlan_scheme, "Show current VLAN configuration")

    parser = _add_project_action(subparsers, "set", _on_set_vlan_scheme, "Set VLAN assigning scheme", with_reason=True)
    parser.add_argument("scheme", choices=VlanScheme.ALL, help="VLAN scheme")
    parser.add_argument("vlan", type=int, help="native VLAN")
    parser.add_argument("-e", "--extra-vlans", help="extra VLANs that should be assigned to each project's host")

    _add_project_action(subparsers, "reset", _on_reset_vlan_scheme,
                        "Reset VLAN scheme (disable VLAN autoconfiguration)", with_reason=True)


def _init_failure_reports(subparsers):
    subparsers = register_subparsers(subparsers, "reports", "Configure failure reports", short="f")
    _add_project_action(subparsers, "get", _on_get_failure_reports, "Show current failure reports configuration")

    # parser for "set" command
    set_parser = _add_project_action(subparsers, "set", _on_set_failure_reports, "Set failure reports configuration",
                                 with_reason=True)
    set_parser.add_argument("queue", help="startrek queue to put tickets to")
    set_parser.add_argument("summary", nargs="?",
                        help="optional summary substring to add to generated summary, normally one word is enough")
    enabled_group = set_parser.add_mutually_exclusive_group()
    enabled_group.add_argument("--enabled", action="store_true", dest="enabled",
                               default=True, help="enable reports [default]")
    enabled_group.add_argument("--disabled", action="store_false", dest="enabled",
                               default=True, help="disable reports")
    set_parser.add_argument("-e", "--extra", action=StoreKeyValuePair, nargs="?",
                            help="key/value pair of extra ticket parameters")

    # parser for "modify" command
    modify_parser = _add_project_action(subparsers, "modify", _on_modify_failure_reports,
                                        "Change failure reports configuration", with_reason=True)
    modify_parser.add_argument("-q", "--queue", help="set new startrek queue to put tickets to")
    modify_parser.add_argument("-s", "--summary", help="set new summary substring to add to generated summary "
                                                       "(normally one word is enough)")
    enabled_group = modify_parser.add_mutually_exclusive_group()
    enabled_group.add_argument("--enabled", action="store_true", dest="enabled", default=None, help="enable reports")
    enabled_group.add_argument("--disabled", action="store_false", dest="enabled", default=None, help="disable reports")

    modify_parser.add_argument("-e", "--extra", action=StoreKeyValuePair, nargs="?",
                               help="set key/value pair of new extra ticket parameters")

    _add_project_action(subparsers, "reset", _on_remove_failure_reports,
                        "Reset failure report settings to global defaults", with_reason=True)


def _init_dns_domain(subparsers):
    subparsers = register_subparsers(subparsers, "dns-domain", "Configure domain for DNS auto configuration",
                                     short="dns")

    parser = _add_project_action(subparsers, "set", _on_set_dns_domain,
                                 "Set domain for DNS auto configuration", with_reason=True)
    parser.add_argument("dns_domain", help="DNS domain name")

    _add_project_action(subparsers, "unset", _on_unset_dns_domain,
                        "Unset domain for DNS auto configuration (disables DNS auto-configuration for the project)",
                        with_reason=True)


def _init_host_shortname_template(subparsers):
    subparsers = register_subparsers(subparsers, "host-shortname-template",
                                     help="Configure custom host shortname template",
                                     short="ht")

    parser = _add_project_action(subparsers, "set", _on_set_custom_host_shortname_template,
                                 "Set custom host shortname template", with_reason=True)
    parser.add_argument("host_shortname_template", help="Custom host shortname template")

    _add_project_action(subparsers, "unset", _on_unset_custom_host_shortname_template,
                        "Unset custom host shortname template", with_reason=True)


def _init_hbf_project_id(subparsers):
    subparsers = register_subparsers(subparsers, "hbf-project-id", "Configure HBF project id for HBF-enabled projects",
                                     short="hbf")

    parser = _add_project_action(subparsers, "set", _on_set_hbf_project_id,
                                 "Set HBF project id for HBF-enabled projects", with_reason=True)
    parser.add_argument("hbf_project_id", help="HBF project id")
    parser.add_argument("--ip-method", choices=MtnIpMethod.ALL,
                        help="Method for generating host-id part of IP-address for HBF/MTN")

    _add_project_action(subparsers, "unset", _on_unset_hbf_project_id,
                        "Unset HBF project id (the project is not HBF-enabled anymore)", with_reason=True)


def _init_bot_project_id(subparsers):
    subparsers = register_subparsers(subparsers, "bot-project-id", "Configure BOT/OEBS project id", short="bot")

    parser = _add_project_action(subparsers, "set", _on_set_bot_project_id, "Set BOT/OEBS project id", with_reason=True)
    parser.add_argument("bot_project_id", help="BOT/OEBS project id")

    _add_project_action(subparsers, "unset", _on_unset_bot_project_id, "Unset BOT project id", with_reason=True)


def _init_automation_plot(subparsers):
    subparsers = register_subparsers(subparsers, "automation-plot", "Configure project automation plot", short="ap")
    _add_project_action(subparsers, "get", _on_get_automation_plot, "Show project automation plot id")

    parser = _add_project_action(
        subparsers, "set", _on_set_automation_plot, "Set project automation plot", with_reason=True)
    parser.add_argument("automation_plot_id", help="Automation Plot id")

    _add_project_action(
        subparsers, "unset", _on_unset_automation_plot, "Unset project automation plot", with_reason=True)


def _init_automation(subparsers):
    automation_subparsers = register_subparsers(subparsers, "automation", "Configure automation", short="w")

    parser = _add_project_action(automation_subparsers, "enable", _on_enable_automation, "Enable automation", with_reason=True)

    parser.add_argument("type", choices=AutomationType.ALL_AUTOMATION_TYPES, default=AutomationType.TYPE_ALL_AUTOMATION,
                        nargs="?", help="[optional] automation type to switch on (default is all types)")
    parser.add_argument("-c", "--credit-time", type=parse_period,
                        help="time to give automation credit for in {} format".format(period_format()))

    for names, metavar, description in (
        (["-u", "--unreachable"],    "UNREACHABLE_FAILURES_CREDIT",    "unreachable check failures"),
        (["-s", "--ssh"],            "SSH_FAILURES_CREDIT",            "ssh check failures"),

        (["--memory"],               "MEMORY_FAILURES_CREDIT",         "memory check failures"),
        (["--disk"],                 "DISK_FAILURES_CREDIT",           "disk check failures"),
        (["--link"],                 "LINK_FAILURES_CREDIT",           "link check failures"),
        (["--cpu"],                  "CPU_FAILURES_CREDIT",            "cpu check failures"),
        (["--overheat"],             "OVERHEAT_CREDIT",                "overheat check failures"),
        (["--bmc"],                  "BMC_FAILURES_CREDIT",            "BMC check failures"),
        (["--reboots"],              "REBOOTS_FAILURES_CREDIT",        "reboots check failures"),
        (["--tainted-kernel"],       "TAINTED_KERNEL_FAILURES_CREDIT", "tainted kernel check failures"),
        (["--cpu-capping"],          "CPU_CAPPING_FAILURES_CREDIT",    "CPU capping check failures"),
        (["--fs-check"],             "FS_CHECK_FAILURES_CREDIT",       "filesystem check failures"),

        (["--checks-missing"],       "CHECKS_MISSING_CREDIT",          "checks missing failures"),
        (["--dead-hosts"],           "DEAD_HOSTS_CREDIT",              "dead hosts"),
        (["-n", "--dns-fixes"],      "DNS_FIXES_CREDIT",               "DNS record fixes"),
    ):
        parser.add_argument(*names, type=int, metavar=metavar,
                            help="credit for maximum number of " + description)

    parser.add_argument("-o", "--other", nargs=2, action=ActionStoreCredit, default={},
                        metavar=("CHECK_NAME", "FAILURE_CREDIT"),
                        help="credit for maximum number of other check failures,"
                             " for example [-o ssh SSH_FAILURES] [-o disk DISK_FAILURES]")

    parser = _add_project_action(automation_subparsers, "disable", _on_disable_automation,
                                 "Disable automation", with_reason=True)
    parser.add_argument("type", choices=AutomationType.ALL_AUTOMATION_TYPES, default=AutomationType.TYPE_ALL_AUTOMATION,
                        nargs="?", help="automation type to switch off (default is all types)")

    automation_limits_subparsers = register_subparsers(automation_subparsers, "limits", "Automation limits")
    _add_project_action(automation_limits_subparsers, "get", _on_get_automation_limits, "Show automation limits")

    parser = _add_project_action(automation_limits_subparsers, "set", _on_set_automation_limits,
                                 "Set automation limits", with_reason=True)
    format_suffix = " in $period:$number format where $period has {} format".format(period_format())

    for names, metavar, description in (
        (["-u", "--unreachable"],    "MAX_UNREACHABLE_FAILURES",      "unreachable check failures"),
        (["-s", "--ssh"],            "MAX_SSH_FAILURES",              "ssh check failures"),

        (["--memory"],               "MAX_MEMORY_FAILURES",           "memory check failures"),
        (["--disk"],                 "MAX_DISK_FAILURES",             "disk check failures"),
        (["--link"],                 "MAX_LINK_FAILURES",             "link check failures"),
        (["--cpu"],                  "MAX_CPU_FAILURES",              "cpu check failures"),
        (["--overheat"],             "MAX_OVERHEATED_HOSTS",          "overheat check failures"),
        (["--bmc"],                  "MAX_BMC_FAILURES",              "BMC check failures"),
        (["--reboots"],              "MAX_REBOOTS_FAILURES",          "reboots check failures"),
        (["--tainted-kernel"],       "MAX_TAINTED_KERNEL_FAILURES",   "tainted kernel check failures"),
        (["--cpu-capping"],          "MAX_CPU_CAPPING_FAILURES",      "CPU capping check failures"),
        (["--fs-check"],             "MAX_FS_CHECK_FAILURES",         "filesystem check failures"),

        (["--checks-missing"],       "MAX_HOSTS_WITH_CHECKS_MISSING", "hosts with checks missing"),
        (["--dead-hosts"],           "MAX_DEAD_HOSTS",                "dead hosts"),
        (["-n", "--dns-fixes"],      "MAX_DNS_FIXES",                 "DNS record fixes"),
    ):
        parser.add_argument(*names, action="append", type=parse_limit,
                            metavar=metavar, help="maximum number of " + description + format_suffix)

    parser.add_argument("-o", "--other", nargs=2, action=ActionAppendLimit, default={},
                        metavar=("CHECK_NAME", "LIMIT_SPEC"),
                        help="maximum number of other check failures {},"
                             " for example [-o ssh 1d:10] [-o disk 1d:20]".format(format_suffix)
                        )


def _init_fsm_handbrake(subparsers):
    subparsers = register_subparsers(subparsers, "fsm-handbrake", "Enable/extend or disable FSM handbrake", short="h")

    parser = _add_project_action(subparsers, "enable", _on_enable_fsm_handbrake,
                                 "Enable/extend FSM handbrake", with_reason=True)

    parser.add_argument("-i", "--ticket-key", help="Startrek ticket key for the incident")

    timeout_group = parser.add_mutually_exclusive_group()
    timeout_group.add_argument("-t", "--timeout", type=int, help="Timeout for FSM handbrake")
    timeout_group.add_argument(
        "-d", "--timeout-time",
        help="Time when FSM handbrake expires in the following format: " + get_supported_time_formats())

    _add_project_action(subparsers, "disable", _on_disable_fsm_handbrake, "Disable FSM handbrake", with_reason=True)


def _init_reboot_via_ssh(subparsers):
    subparsers = register_subparsers(subparsers, "reboot-via-ssh", "Enable/disable rebooting by SSH", short="rs")

    _add_project_action(subparsers, "enable", _on_enable_reboot_via_ssh,
                        "Enable rebooting via SSH", with_reason=True)

    _add_project_action(subparsers, "disable", _on_disable_reboot_via_ssh,
                        "Disable rebooting via SSH", with_reason=True)


def _init_notifications(subparsers):
    notifications_subparsers = register_subparsers(subparsers, "notifications", "Configure notifications")
    recipients_subparsers = register_subparsers(notifications_subparsers, "recipients", "Configure recipients")
    _add_project_action(recipients_subparsers, "get", _on_get_recipients, "Show notification recipients")

    for action in WalleClient.get_api_actions():
        parser = _add_project_action(recipients_subparsers, action, _get_on_modify_recipients_handler(action),
                                     "{} recipients for severity".format(action.capitalize()), with_reason=True)

        for severity in _NOTIFICATION_SEVERITIES:
            parser.add_argument("-" + severity[0], "--" + severity, metavar="EMAILS",
                                help="a comma-separated list of emails for {} severity".format(severity.upper()))


def _init_host_profiling_config(subparsers):
    subparsers = register_subparsers(subparsers, "profiling", "Configure host profiling")
    _add_project_action(subparsers, "get", _on_get_profiling_config, "Show host profiling configuration")
    parser = _add_project_action(subparsers, "set", _on_set_profiling_config, "Set host profiling configuration", with_reason=True)
    parser.add_argument("-p", "--profile", required=True, help="default Einstellung profile")
    parser.add_argument("-t", "--profile-tags", help="Einstellung tags to assign to host before profiling")


def _init_host_provisioner_config(subparsers):
    subparsers = register_subparsers(subparsers, "provisioner", "Configure host deploying", short="d")
    _add_project_action(subparsers, "get", _on_get_provisioner_config, "Show host provisioning configuration")
    parser = _add_project_action(subparsers, "set", _on_set_provisioner_config, "Set host provisioning configuration", with_reason=True)
    parser.add_argument("-p", "--provisioner", required=True, help="default provisioner")
    parser.add_argument("-c", "--deploy-config", required=True, help="default deploy config")
    parser.add_argument("-dcp", "--deploy-config-policy", help="Policy to (optionally) alter deploy config")
    parser.add_argument("-t", "--deploy-tags", help="Deploy tags to assign to host before deploying")
    parser.add_argument("-n", "--deploy-network", choices=DeployNetwork.ALL,
                        help="deploy host in project or service network")


def _add_project_action(subparsers, name, handler, description, short=None, with_reason=False):
    parser = register_parser(subparsers, name, handler, description, short=short, with_reason=with_reason)
    parser.add_argument("id", help="project ID")
    return parser


def _add_restrictions_argument(parser):
    parser.add_argument("-r", "--default-host-restrictions",
                        help="a comma-separated list of default host restrictions ({})".format(
                            ", ".join(Restriction.ALL)))


@render_result
def _on_get_profiling_config(client, args):
    profiling_config = client.get_project(args.id, fields=["profile", "profile_tags"])
    return {"profiling_config": profiling_config}


def _on_set_profiling_config(client, args):
    question_user(args, "Are you sure you want to modify profiling configuration for project '{}'?", args.id)
    client.set_profiling_config(args.id, profile=args.profile, profile_tags=parse_arg_list(args.profile_tags))


@render_result
def _on_get_provisioner_config(client, args):
    provisioner_config = client.get_project(
        args.id, fields=["provisioner", "deploy_config", "deploy_config_policy", "deploy_tags", "deploy_network"])

    return {"provisioner_config": provisioner_config}


def _on_set_provisioner_config(client, args):
    question_user(args, "Are you sure you want to modify provisioner configuration for project '{}'?", args.id)
    client.set_provisioner_config(args.id,
                                  provisioner=args.provisioner,
                                  deploy_config=args.deploy_config,
                                  deploy_config_policy=args.deploy_config_policy,
                                  deploy_tags=parse_arg_list(args.deploy_tags),
                                  deploy_network=args.deploy_network)


@render_result
def _on_get(client, args):
    project = client.get_project(args.id, fields=_TABLE_VIEW.get_column_fields() + ["owners", "owned_vlans", "automation_limits", "notifications"])
    project.pop("id", None)
    project.setdefault("default_host_restrictions", [])
    return {"project": project}


def _on_list(client, args):
    columns, fields = _TABLE_VIEW.parse_column_list(args.columns)
    projects = client.get_projects(fields=fields, tags=args.tags)["result"]

    def status(project, label):
        try:
            return "enabled" if project[label + "_automation"]["enabled"] else "disabled"
        except KeyError:
            return "disabled"

    for project in projects:
        for key in "default_host_restrictions", "profile_tags", "deploy_tags":
            if key in project:
                project[key] = ",".join(project[key])

        if "cms_settings" in project:
            result = ""
            for cms in project["cms_settings"]:
                result += ",".join("{}={}".format(key, value) for key, value in cms.items()) + ";"
            project["cms_settings"] = result

        if "tags" in project:
            project["tags"] = ",".join("#" + tag for tag in project["tags"])

        if "extra_vlans" in project:
            project["extra_vlans"] = ",".join(str(vlan) for vlan in project["extra_vlans"])

        if "automation" in columns:
            project["automation"] = " / ".join("{}: {}".format(label, status(project, label))
                                               for label in ("healing", "dns"))
        if "fsm_handbrake" in project:
            project["fsm_handbrake"] = _fmt_fsm_handbrake(project["fsm_handbrake"])

    _TABLE_VIEW.render("projects", projects, columns, args.batch)


def _on_add(client, args):
    question_user(args, "Are you sure you want to add '{}' project with '{}' name?", args.id, args.name)
    client.add_project(
        args.id, args.name, tags=parse_arg_list(args.tags), owners=parse_arg_list(args.owners),
        dns_domain=args.dns_domain, host_shortname_template=args.host_shortname_template,
        hbf_project_id=args.hbf_project_id, ip_method=args.ip_method,
        bot_project_id=int(args.bot_project_id) if args.bot_project_id else None,
        provisioner=args.provisioner, deploy_config=args.deploy_config,
        deploy_tags=parse_arg_list(args.deploy_tags), deploy_network=args.deploy_network,
        default_host_restrictions=parse_arg_list(args.default_host_restrictions), reason=args.reason,
        certificate_deploy=args.certificate_deploy, reboot_via_ssh=args.reboot_via_ssh)


def _on_modify(client, args):
    question_user(args, "Are you sure you want to modify '{}' project?", args.id)
    client.modify_project(args.id, name=args.name, tags=parse_arg_list(args.tags),
                          hbf_project_id=args.hbf_project_id, ip_method=args.ip_method,
                          host_shortname_template=args.host_shortname_template,
                          bot_project_id=int(args.bot_project_id) if args.bot_project_id else None,
                          default_host_restrictions=parse_arg_list(args.default_host_restrictions),
                          certificate_deploy=args.certificate_deploy,
                          reason=args.reason)


def _on_clone(client, args):
    question_user(args, "Are you sure you want to clone project '{}' into '{}' with name '{}'?",
                  args.id, args.clone_id, args.clone_name)
    client.clone_project(args.id, args.clone_id, args.clone_name, reason=args.reason)


@render_result
def _on_get_owners(client, args):
    owners = client.get_project(args.id, fields=["owners"])["owners"]
    return {"owners": owners}


def _get_on_modify_owners_handler(action):
    def on_modify_owners(client, args):
        question_user(args, "Are you sure you want to modify '{}' project owners?", args.id)
        client.modify_project_owners(args.id, action, parse_arg_list(args.logins), reason=args.reason)

    return on_modify_owners


@render_result
def _on_get_owned_vlans(client, args):
    owned_vlans = client.get_project(args.id, fields=["owned_vlans"])["owned_vlans"]
    return {"owned_vlans": owned_vlans}


def _get_on_modify_owned_vlans_handler(action):
    def on_modify_owned_vlans(client, args):
        question_user(args, "Are you sure you want to modify VLANs owned by '{}' project?", args.id)
        client.modify_project_owned_vlans(args.id, action, parse_arg_list(args.vlans, type=int), reason=args.reason)

    return on_modify_owned_vlans


def _get_on_modify_tags_handler(action):
    def _on_modify_tags(client, args):
        question_user(args, "Are you sure you want to {} '{}' project's tags?", action, args.id)
        client.modify_project_tags(args.id, action, parse_arg_list(args.tags), reason=args.reason)

    return _on_modify_tags


def _on_get_vlan_scheme(client, args):
    project = client.get_project(args.id, fields=["vlan_scheme", "native_vlan", "extra_vlans"])

    if "vlan_scheme" in project:
        print_object("vlan_configuration", project, as_json=args.json)
    else:
        print("The project has no assigned VLAN scheme.")


def _on_set_vlan_scheme(client, args):
    _question_user_on_project_action(args, "set VLAN scheme")
    client.set_project_vlan_scheme(args.id, args.scheme, args.vlan,
                                   extra_vlans=parse_arg_list(args.extra_vlans, type=int), reason=args.reason)


def _on_reset_vlan_scheme(client, args):
    _question_user_on_project_action(args, "reset VLAN scheme")
    client.reset_project_vlan_scheme(args.id, reason=args.reason)


@render_result
def _on_get_failure_reports(client, args):
    report_parameters = client.get_project(args.id, fields=["reports"]).get("reports", {})
    return {"report_parameters": report_parameters}


def _on_set_failure_reports(client, args):
    question_user(args, "Are you sure you want to set failure report parameters for '{}' project?", args.id)
    client.set_failure_report_parameters(args.id, enabled=args.enabled, queue=args.queue, summary=args.summary,
                                         extra=split_comma_separated_values_in_dict(args.extra), reason=args.reason)


def _on_modify_failure_reports(client, args):
    question_user(args, "Are you sure you want to update failure report parameters for '{}' project?", args.id)
    client.modify_failure_report_parameters(args.id, enabled=args.enabled, queue=args.queue, summary=args.summary,
                                            extra=split_comma_separated_values_in_dict(args.extra), reason=args.reason)


def _on_remove_failure_reports(client, args):
    question_user(args, "Are you sure you want to reset failure report parameters for '{}' project?", args.id)
    client.remove_failure_report_parameters(args.id, reason=args.reason)


def _on_set_dns_domain(client, args):
    _SIMPLE_FQDN_RE = re.compile(r"(?:[a-z0-9]+(?:-[a-z0-9]+)*\.)+[a-z]{2,}(?::[0-9]+)?")
    if not re.match(_SIMPLE_FQDN_RE, args.dns_domain):
        raise Error("Invalid domain: {}.", args.dns_domain)

    _question_user_on_project_action(args, "set DNS domain to {}".format(args.dns_domain))
    client.set_project_dns_domain(args.id, args.dns_domain, reason=args.reason)


def _on_unset_dns_domain(client, args):
    _question_user_on_project_action(args, "unset dns domain")
    client.unset_project_dns_domain(args.id, reason=args.reason)


def _on_set_custom_host_shortname_template(client, args):
    _question_user_on_project_action(
        args, "set custom host shortname template to '{}'".format(args.host_shortname_template))

    client.set_project_host_shortname_template(args.id, args.host_shortname_template, reason=args.reason)


def _on_unset_custom_host_shortname_template(client, args):
    _question_user_on_project_action(args, "unset custom host shortname template")
    client.unset_project_host_shortname_template(args.id, reason=args.reason)


def _on_set_hbf_project_id(client, args):
    try:
        int(args.hbf_project_id, 16)
    except ValueError:
        raise Error("Invalid value for hbf_project_id: {}, need a hexadecimal number.", args.hbf_project_id)

    _question_user_on_project_action(args, "set HBF project id to {}".format(args.hbf_project_id))
    client.set_project_hbf_project_id(
        args.id, args.hbf_project_id,
        ip_method=args.ip_method,
        reason=args.reason)


def _on_unset_hbf_project_id(client, args):
    _question_user_on_project_action(args, "unset HBF project id")
    client.unset_project_hbf_project_id(args.id, reason=args.reason)


def _on_set_bot_project_id(client, args):
    try:
        bot_project_id = int(args.bot_project_id)
    except ValueError:
        raise Error("Invalid value for bot_project_id: {}, need an integer.", args.bot_project_id)

    _question_user_on_project_action(args, "set BOT/OEBS project id to {}".format(bot_project_id))
    client.set_project_bot_project_id(args.id, bot_project_id, reason=args.reason)


def _on_unset_bot_project_id(client, args):
    _question_user_on_project_action(args, "unset BOT project id")
    client.unset_project_bot_project_id(args.id, reason=args.reason)


def _on_enable_fsm_handbrake(client, args):
    _question_user_on_project_action(args, "enable FSM handbrake")
    client.enable_project_fsm_handbrake(
        args.id, timeout=args.timeout, timeout_time=parse_time(args.timeout_time, future_period=True),
        ticket_key=args.ticket_key, reason=args.reason
    )


def _on_disable_fsm_handbrake(client, args):
    _question_user_on_project_action(args, "disable FSM handbrake")
    client.disable_project_fsm_handbrake(args.id, reason=args.reason)


def _on_enable_reboot_via_ssh(client, args):
    _question_user_on_project_action(args, "enable reboot via SSH")
    client.modify_project_enable_reboot_via_ssh(args.id, reason=args.reason)


def _on_disable_reboot_via_ssh(client, args):
    _question_user_on_project_action(args, "disable reboot via SSH")
    client.modify_project_disable_reboot_via_ssh(args.id, reason=args.reason)


@render_result
def _on_get_automation_plot(client, args):
    plot = client.get_project(args.id, fields=["automation_plot_id"])
    return {"automation_plot": plot}


def _on_set_automation_plot(client, args):
    _question_user_on_project_action(args, "set automation plot to {}".format(args.automation_plot_id))
    client.set_project_automation_plot(args.id, args.automation_plot_id, reason=args.reason)


def _on_unset_automation_plot(client, args):
    _question_user_on_project_action(args, "unset automation plot")
    client.unset_project_automation_plot(args.id, reason=args.reason)


@render_result
def _on_enable_automation(client, args):
    with_credit = any(credit is not None for credit in (
        args.dns_fixes, args.unreachable, args.ssh, args.memory, args.disk, args.link, args.cpu, args.reboots,
        args.tainted_kernel, args.fs_check, args.dead_hosts)) or args.other

    if with_credit and args.credit_time is None:
        raise Error("Credit time must be specified if you want to give a credit for automated actions.")

    all_limits = filter_none(dict(
        max_unreachable_failures=args.unreachable,
        max_ssh_failures=args.ssh,
        max_memory_failures=args.memory,
        max_disk_failures=args.disk,
        max_link_failures=args.link,
        max_cpu_failures=args.cpu,
        max_overheat_failures=args.overheat,
        max_bmc_failures=args.bmc,
        max_reboots_failures=args.reboots,
        max_tainted_kernel_failures=args.tainted_kernel,
        max_cpu_capping_failures=args.cpu_capping,
        max_fs_check_failures=args.fs_check,
        max_checks_missing_failures=args.checks_missing,
        max_dead_hosts=args.dead_hosts,
        max_dns_fixes=args.dns_fixes,
    ))

    if all_limits.keys() & args.other.keys():
        raise Error(
            "You have specified multiple credit values for limit(s) '{}'.",
            "', '".join(sorted(all_limits.keys() & args.other.keys()))
        )
    all_limits.update(args.other)

    _question_user_on_project_action(args, "enable automation")

    res = client.enable_automation(
        args.id, args.type, reason=args.reason,
        credit_time=args.credit_time,
        **all_limits)["result"]
    return {"result": res}


@render_result
def _on_disable_automation(client, args):
    _question_user_on_project_action(args, "disable automation")
    res = client.disable_automation(args.id, args.type, reason=args.reason)["result"]
    return {"result": res}


@render_result
def _on_get_automation_limits(client, args):
    project = client.get_project(args.id, fields=["automation_limits"])
    return {"automation_limits": project["automation_limits"]}


def _on_set_automation_limits(client, args):
    all_limits = filter_none(dict(
        max_unreachable_failures=args.unreachable,
        max_ssh_failures=args.ssh,
        max_memory_failures=args.memory,
        max_disk_failures=args.disk,
        max_link_failures=args.link,
        max_cpu_failures=args.cpu,
        max_overheat_failures=args.overheat,
        max_bmc_failures=args.bmc,
        max_reboots_failures=args.reboots,
        max_tainted_kernel_failures=args.tainted_kernel,
        max_cpu_capping_failures=args.cpu_capping,
        max_fs_check_failures=args.fs_check,
        max_checks_missing_failures=args.checks_missing,
        max_dead_hosts=args.dead_hosts,
        max_dns_fixes=args.dns_fixes,
    ))

    for limit_name, limits in args.other.items():
        all_limits.setdefault(limit_name, []).extend(limits)

    _question_user_on_project_action(args, "modify automation limits")
    client.modify_project_automation_limits(args.id, reason=args.reason, **all_limits)


@render_result
def _on_get_recipients(client, args):
    project = client.get_project(args.id, fields=["notifications.recipients"])
    recipients = project["notifications"]["recipients"]
    return {"recipients": recipients}


def _get_on_modify_recipients_handler(action):
    def on_modify_recipients(client, args):
        _question_user_on_project_action(args, "modify notification recipients")
        recipients_kwargs = {severity: parse_arg_list(getattr(args, severity)) for severity in _NOTIFICATION_SEVERITIES}
        client.modify_project_notification_recipients(args.id, action, reason=args.reason, **recipients_kwargs)

    return on_modify_recipients


def _on_remove(client, args):
    question_user(args, "Are you sure you want to remove '{}' project?", args.id)
    client.remove_project(args.id, reason=args.reason)


@render_result
def _on_get_role_members(client, args):
    members = client.get_project_role_members(args.id, args.role)
    return members


def _on_request_add_role_member(client, args):
    client.request_add_project_role_member(args.id, args.role, args.member)


def _on_request_remove_role_member(client, args):
    client.request_remove_project_role_member(args.id, args.role, args.member)


def _question_user_on_project_action(args, action):
    question_user(args, "Are you sure you want to {} for '{}' project?", action, args.id)


def _fmt_fsm_handbrake(fsm_handbrake_obj):
    if "ticket_key" in fsm_handbrake_obj:
        reason_str = " (see {})".format(fsm_handbrake_obj["ticket_key"])
    elif "reason" in fsm_handbrake_obj:
        reason_str = ": {}".format(fsm_handbrake_obj["reason"])
    else:
        reason_str = ""

    return "Enabled by {issuer} till {timeout_time}{reason}".format(
        issuer=fsm_handbrake_obj["issuer"],
        timeout_time=format_time(fsm_handbrake_obj["timeout_time"]),
        reason=reason_str,
    )


def _init_repair_request_severity(subparsers):
    subparsers = register_subparsers(subparsers, "repair-request-severity",
                                     "Configure repair request severity for hosts in EINE & BOT",
                                     short="severity")

    parser = _add_project_action(subparsers, "set", _on_set_repair_request_severity,
                                 "Set repair request severity", with_reason=True)
    parser.add_argument("request_severity", help="Repair request severity for hosts in EINE & BOT")


def _on_set_repair_request_severity(client, args):
    _SEVERITIES = ["high", "medium", "low"]
    if args.request_severity not in _SEVERITIES:
        raise Error("Invalid severity: {}. Must be one of: {}", args.request_severity, _SEVERITIES)

    _question_user_on_project_action(args, "set repair request severity to {}".format(args.request_severity))
    client.set_project_repair_request_severity(args.id, args.request_severity, reason=args.reason)


def _init_modify_project_cms(subparsers):
    subparsers = register_subparsers(subparsers, "modify-project-cms",
                                     "Modify project cms settings",
                                     short="modify-cms")

    parser = _add_project_action(subparsers, "add", _on_add_project_cms_settings,
                                 "Add project cms settings", with_reason=True)
    _add_cms_arguments(parser)
    parser = _add_project_action(subparsers, "replace", _on_replace_project_cms_settings,
                                 "Replace project cms settings", with_reason=True)
    _add_cms_arguments(parser)


def _on_add_project_cms_settings(client, args):
    cms_settings = _cms_fields_conversion(client.get_project(args.id, fields=["cms_settings"])["cms_settings"])
    cms_data = _parse_cms(args)
    cms_settings.append(cms_data)
    client.set_project_cms(args.id, _format_response(cms_settings), reason=args.reason)


def _on_replace_project_cms_settings(client, args):
    cms_settings = []
    cms_data = _parse_cms(args)
    cms_settings.append(cms_data)
    client.set_project_cms(args.id, _format_response(cms_settings), reason=args.reason)


def _add_cms_arguments(parser):
    parser.add_argument("-c", "--cms", help="CMS URL or 'default' to use default CMS")
    parser.add_argument("-v", "--cms-api-version", choices=CmsApiVersion.ALL,
                        help="CMS API version supported by the CMS (custom CMS only)")
    parser.add_argument("--max-busy-hosts",
                        type=int,
                        help="Default CMS only! Maximum number of hosts that default CMS will allow to process "
                             "concurrently")
    parser.add_argument("--tvm-app-id",
                        type=int,
                        help="Non-default and not YP CMS only! TVM app id of CMS, required when used not by Wall-e "
                             "administrators")
    parser.add_argument("--temporary_unreachable_enabled",
                        type=bool,
                        help="Shows if 'temporary-unreachable' is properly implemented, default is false",
                        default=False)


def _parse_cms(args):
    if args.cms is None:
        if args.max_busy_hosts is not None or args.cms_api_version is not None:
            raise Error("Either specify cms url or don't specify cms parameters at all")

    elif args.cms == "default":
        if args.max_busy_hosts is None:
            raise Error("Specify --max-busy-hosts for default CMS")

        if args.cms_api_version is not None:
            raise Error("--cms-api-version is not allowed for default CMS")

        if args.tvm_app_id is not None:
            raise Error("--tvm-app-id is not allowed for default CMS")

    else:
        if args.max_busy_hosts is not None:
            raise Error("--max-busy-hosts is not allowed for non-default CMS")

        if args.cms_api_version is None:
            raise Error("--cms-api-version is required for non-default CMS")

    return filter_none({
        "url": args.cms,
        "max_busy_hosts": args.max_busy_hosts,
        "api_version": args.cms_api_version,
        "tvm_app_id": args.tvm_app_id,
        "temporary_unreachable_enabled": args.temporary_unreachable_enabled
    })


def _cms_fields_conversion(cms_settings):
    result = []
    for setting in cms_settings:
        converted_setting = {}
        for key, val in setting.items():
            if key == "cms":
                converted_setting["url"] = val
            elif key == "cms_api_version":
                converted_setting["api_version"] = val
            elif key == "cms_max_busy_hosts":
                converted_setting["max_busy_hosts"] = val
            elif key == "cms_tvm_app_id":
                converted_setting["tvm_app_id"] = val
            else:
                converted_setting[key] = val
        result.append(converted_setting)
    return result


def _format_response(settings):
    DEFAULT_SCHEMA_KEYS = {"url", "max_busy_hosts"}
    CUSTOM_CMS_KEYS = {"url", "api_version", "temporary_unreachable_enabled", "tvm_app_id"}
    result = []

    def _filter_dict_by_keys(dct, filter_keys):
        result = {}
        for key in dct.keys():
            if key in filter_keys:
                result[key] = dct[key]
        return result

    for setting in settings:
        if setting["url"] == "default":
            result.append(_filter_dict_by_keys(setting, DEFAULT_SCHEMA_KEYS))
        else:
            result.append(_filter_dict_by_keys(setting, CUSTOM_CMS_KEYS))
    return result
