"""Scenario management."""

from walle_cli.common import TableView, Column, register_subparsers, register_parser, drop_none, question_user


class ScenarioTypes:
    NOOP = "noop"
    WAIT = "wait"
    MAINTENANCE = "switch-to-maintenance"
    SWITCH_PROJECT = "switch-project"
    HOSTS_ADD = "hosts-add"
    HOSTS_TRANSFER = "hosts-transfer"
    ITDC_MAINTENANCE = "itdc-maintenance"


_SCENARIO_TABLE_VIEW = TableView(
    default_columns=(
        Column("scenario_id", "ID"),
        Column("name", "Name"),
        Column("ticket_key", "Ticket"),
        Column("scenario_type", "Type"),
        Column("status", "Status"),
        Column("issuer", "Issuer"),
    ),
    other_columns=(
        Column("hosts", "Number of hosts"),
        Column("script_args", "Scenario's specific arguments"),
        Column("labels", "Labels"),
        Column("creation_time", "Creation time")
    )
)


_HOSTS_TABLE_VIEW = TableView(
    default_columns=(
        Column("inv", "Inventory number"),
        Column("status", "Status"),
        Column("state", "State")
    )
)


_ERRORS_TABLE_VIEW = TableView(
    default_columns=(
        Column("id", "Error id"),
        Column("is_visible", "Is visible to user?"),
        Column("repeats", "Repeats count"),
        Column("retry_period", "Retry period"),
        Column("type", "Error type"),
        Column("exc_info", "Error info")
    )
)


def init(subparsers):
    subparsers = register_subparsers(subparsers, "scenarios", "Scenario management", short="sc")
    _init_list(subparsers)
    _init_add_scenario(subparsers)
    _init_show_hosts(subparsers)
    _init_start_scenario(subparsers)
    _init_cancel_scenario(subparsers)
    _init_pause_scenario(subparsers)
    _init_itdc_maintenance_scenario(subparsers)
    _init_scenario_errors(subparsers)


def _init_show_hosts(subparsers):
    parser = register_parser(subparsers, "show-hosts", _show_hosts,
                             short="h", help="Show host info of selected scenario")
    parser.add_argument("scenario", help="Scenario ID")
    parser.add_argument("-C", "--columns", help="a comma-separated list of columns to output ({})".format(
        ", ".join(_HOSTS_TABLE_VIEW.get_column_ids())))


def _show_hosts(client, args):
    response = client.get_scenario(args.scenario)
    columns, _ = _HOSTS_TABLE_VIEW.parse_column_list(args.columns)
    _display_host_info(response, columns, args.batch)


def _display_host_info(scenario, columns, batch):
    result = []
    for entry in scenario["hosts"]:
        result.append(
            {
                "inv": entry["inv"],
                "status": entry["status"],
            }
        )
    _HOSTS_TABLE_VIEW.render("hosts", result, columns, batch, total=len(result))


def _init_add_scenario(subparsers):
    subparsers = register_subparsers(subparsers, "add", "Add scenario")
    _init_add_noop_scenario(subparsers)
    _init_add_wait_scenario(subparsers)
    _init_add_maintenance_scenario(subparsers)
    _init_add_hosts_add_scenario(subparsers)
    _init_add_switch_project_scenario(subparsers)


def _add_common_scenario_params(parser):
    parser.add_argument("--name", help="Name of the scenario")
    parser.add_argument("--schedule-type", help="Schedule type")
    parser.add_argument("--ticket-key", help="Key of related ticket")
    parser.add_argument("--responsible", help="Login of responsible user")
    parser.add_argument("--labels", nargs="?",
                        help="Comma-separated list of labels in format <key>=<value>,<key2>=<value2>")
    parser.add_argument("--autostart", action="store_true", help="start scenario immediately after creation")
    parser.add_argument("--hosts", nargs="?", help="Comma-separated list of hosts assigned to scenario")


def _init_add_noop_scenario(subparsers):
    parser = register_parser(subparsers, "noop", _on_add_noop,
                             help="Create scenario to keep hosts in (noop)", with_reason=True)
    _add_common_scenario_params(parser)


def _init_add_maintenance_scenario(subparsers):
    parser = register_parser(subparsers, "maintenance", _on_add_maintenance,
                             help="Switch hosts to maintenance using scenario", with_reason=True)
    _add_common_scenario_params(parser)

    parser.add_argument("--timeout", type=int, nargs="?", help="The time until the end of the scenario")
    parser.add_argument("--switch", nargs="?", help="The switch to collect hosts from")


def _init_add_wait_scenario(subparsers):
    parser = register_parser(subparsers, "wait", _on_add_wait,
                             help="Wait for hosts to be ready using scenario", with_reason=True)
    _add_common_scenario_params(parser)

    parser.add_argument("--target-project-id", help="Id of target project")


def _init_add_hosts_add_scenario(subparsers):
    parser = register_parser(subparsers, "add_hosts", _on_add_hosts_add,
                             help="Add hosts using scenario", with_reason=True)
    _add_common_scenario_params(parser)

    parser.add_argument("--target-project-id", help="Id of target project")


def _init_add_switch_project_scenario(subparsers):
    parser = register_parser(subparsers, "switch_project", _on_add_switch_project,
                             help="Switch hosts' project using scenario", with_reason=True)
    _add_common_scenario_params(parser)

    parser.add_argument("--target-project-id", help="Id of target project")


def _hosts_list_from_args(hosts_arg):
    return [int(host.strip()) for host in hosts_arg.split(",")] if hosts_arg else None


def _labels_dict_from_args(labels_arg):
    labels = {}
    if not labels_arg:
        return

    for pair in labels_arg.split(","):
        key, value = pair.split("=")
        labels[key] = value
    return labels


def _on_add_maintenance(client, args):
    script_args = {
        "timeout": args.timeout,
        "switch": args.switch,
        "schedule_type": args.schedule_type
    }
    client.add_scenario(name=args.name, scenario_type=ScenarioTypes.MAINTENANCE, script_args=drop_none(script_args),
                        labels=_labels_dict_from_args(args.labels), hosts=_hosts_list_from_args(args.hosts),
                        ticket_key=args.ticket_key, autostart=args.autostart, reason=args.reason)


def _on_add_noop(client, args):
    script_args = {
        "schedule_type": args.schedule_type
    }
    client.add_scenario(name=args.name, scenario_type=ScenarioTypes.NOOP, autostart=args.autostart,
                        labels=_labels_dict_from_args(args.labels), script_args=drop_none(script_args),
                        hosts=_hosts_list_from_args(args.hosts), ticket_key=args.ticket_key, reason=args.reason)


def _on_add_wait(client, args):
    script_args = {
        "target_project_id": args.target_project_id,
        "responsible": args.responsible,
        "schedule_type": args.schedule_type
    }
    client.add_scenario(name=args.name, scenario_type=ScenarioTypes.WAIT, ticket_key=args.ticket_key,
                        labels=_labels_dict_from_args(args.labels),
                        hosts=_hosts_list_from_args(args.hosts), script_args=drop_none(script_args),
                        autostart=args.autostart, reason=args.reason)


def _on_add_hosts_add(client, args):
    script_args = {
        "target_project_id": args.target_project_id,
        "responsible": args.responsible,
        "schedule_type": args.schedule_type
    }
    client.add_scenario(name=args.name, scenario_type=ScenarioTypes.HOSTS_ADD,
                        ticket_key=args.ticket_key, labels=_labels_dict_from_args(args.labels),
                        hosts=_hosts_list_from_args(args.hosts),
                        script_args=drop_none(script_args), autostart=args.autostart, reason=args.reason)


def _on_add_switch_project(client, args):
    script_args = {
        "target_project_id": args.target_project_id,
        "responsible": args.responsible,
        "schedule_type": args.schedule_type
    }
    client.add_scenario(name=args.name, scenario_type=ScenarioTypes.SWITCH_PROJECT,
                        ticket_key=args.ticket_key, labels=_labels_dict_from_args(args.labels),
                        hosts=_hosts_list_from_args(args.hosts),
                        script_args=drop_none(script_args), autostart=args.autostart, reason=args.reason)


def _init_list(subparsers):
    parser = register_parser(subparsers, "list", _on_list, "Show all scenarios")
    parser.add_argument("--scenario", help="filter by scenario id")
    parser.add_argument("--name", help="filter by name")
    parser.add_argument("--scenario-type", help="filter by scenario type")
    parser.add_argument("--ticket-key", help="filter by ticket key")
    parser.add_argument("--issuer", help="filter by issuer")
    parser.add_argument("--status", help="filter by scenario status")
    parser.add_argument("--limit", type=int, default=100,
                        help="maximum number of returned hosts (default is 100)")
    parser.add_argument("--show-all", action="store_true", help="show all scenarios")
    parser.add_argument("--columns", help="a comma-separated list of columns to output ({})".format(
        ", ".join(_SCENARIO_TABLE_VIEW.get_column_ids())))


def _on_list(client, args):
    params = {
        "scenario_id": args.scenario,
        "name":  args.name,
        "scenario_type": args.scenario_type,
        "issuer": args.issuer,
        "ticket_key": args.ticket_key,
        "status": args.status,
        "limit": args.limit if not args.show_all else None
    }
    columns, fields = _SCENARIO_TABLE_VIEW.parse_column_list(args.columns)
    scenarios = list(client.iter_scenarios(fields=fields, **params))
    _display_scenarios(scenarios, columns, args.batch, total=len(scenarios))


def _display_scenarios(scenarios, columns, batch, total):
    if "hosts" in columns:
        for scenario in scenarios:
            scenario["hosts"] = len(scenario["hosts"])

    _SCENARIO_TABLE_VIEW.render("hosts", scenarios, columns, batch, total=total)


def _init_start_scenario(subparsers):
    parser = register_parser(subparsers, "start", _on_start, "Start scenario", short="s", with_reason=True)
    parser.add_argument("scenario_id", help="Scenario id")


def _on_start(client, args):
    client.start_scenario(args.scenario_id, reason=args.reason)


def _init_cancel_scenario(subparsers):
    parser = register_parser(subparsers, "cancel", _on_cancel_scenario, "Cancel scenario", short="c", with_reason=True)
    parser.add_argument("scenario_id", help="Scenario id")


def _on_cancel_scenario(client, args):
    scenario = client.get_scenario(args.scenario_id, fields=("scenario_id", "name", "status"))
    question_user(args, "Are you sure you want to cancel scenario '{}' #{} in '{}' status?",
                  scenario["name"], args.scenario_id, scenario["status"])
    client.cancel_scenario(scenario_id=args.scenario_id, reason=args.reason)


def _init_pause_scenario(subparsers):
    parser = register_parser(subparsers, "pause", _on_pause_scenario, "Pause scenario", short="p", with_reason=True)
    parser.add_argument("scenario_id", help="Scenario id")


def _on_pause_scenario(client, args):
    scenario = client.get_scenario(args.scenario_id, fields=("scenario_id", "name", "status"))
    question_user(args, "Are you sure you want to pause scenario '{}' #{} in '{}' status?",
                  scenario["name"], args.scenario_id, scenario["status"])
    client.pause_scenario(scenario_id=args.scenario_id, reason=args.reason)


def _init_itdc_maintenance_scenario(subparsers):
    parser = register_parser(subparsers, "itdc", _on_itdc_maintenance_scenario, "start recovering from ITDC maintenance scenario",
                             short="i", with_reason=True)
    parser.add_argument("scenario_id", help="Scenario id")


def _on_itdc_maintenance_scenario(client, args):
    scenario = client.get_scenario(args.scenario_id, fields=("scenario_id", "name", "status"))
    question_user(args, "Are you sure you want to start recovering from ITDC maintenance for scenario '{}' #{} in '{}' status?",
                  scenario["name"], args.scenario_id, scenario["status"])
    client.start_itdc_maintenance(scenario_id=args.scenario_id, reason=args.reason)


def _init_scenario_errors(subparsers):
    parser = register_parser(subparsers, "errors", _on_show_scenario_errors, "show scenario errors", short="e")
    parser.add_argument("scenario_id", help="Scenario id")
    parser.add_argument("-C", "--columns", help="a comma-separated list of columns to output ({})".format(
        ", ".join(_ERRORS_TABLE_VIEW.get_column_ids())))


def _on_show_scenario_errors(client, args):
    scenario = client.get_scenario(args.scenario_id, fields=("scenario_id", "name", "status", "errors"))
    columns, _ = _ERRORS_TABLE_VIEW.parse_column_list(args.columns)
    _display_errors_info(scenario, columns, args.batch)


def _display_errors_info(scenario, columns, batch):
    result = []
    for entry in scenario["errors"].values():
        row = {column: entry[column] for column in columns}
        result.append(row)
    _ERRORS_TABLE_VIEW.render("errors", result, columns, batch, total=len(result))
