"""Settings management."""

import sys

from walle_cli.common import register_subparsers, register_parser, question_user


def init(subparsers):
    subparsers = register_subparsers(subparsers, "settings", "Settings management")

    # emergency switches. Hide from the top level but keep easily available.
    register_parser(subparsers, "disable_automation", _on_automation(disable=True),
                    "Disable automation globally", with_reason=True)
    register_parser(subparsers, "enable_automation", _on_automation(disable=False),
                    "Disable automation globally", with_reason=True)

    register_parser(subparsers, "list", _on_list, "List available settings")

    parser = register_parser(subparsers, "set", _on_set, "Set setting", with_reason=True)

    healing_group = parser.add_mutually_exclusive_group()
    healing_group.add_argument("--disable-healing", action="store_false", dest="healing", default=None,
                               help="disable automated host healing globally")
    healing_group.add_argument("--enable-healing", action="store_true", dest="healing", default=None,
                               help="enable automated host healing globally (does not affect project's settings)")

    dns_group = parser.add_mutually_exclusive_group()
    dns_group.add_argument("--disable-dns-automation", action="store_false", dest="dns_automation", default=None,
                           help="disable automated dns recovery globally")
    dns_group.add_argument("--enable-dns-automation", action="store_true", dest="dns_automation", default=None,
                           help="enable automated dns recovery globally (does not affect project's settings)")

    parser.add_argument("--invalid-hosts-limit", action="store", type=int, dest="inventory_invalid_hosts_limit",
                        help="Bump invalid hosts limit for inventory sync")

    fsm_handbrake = parser.add_mutually_exclusive_group()
    fsm_handbrake.add_argument("--enable-fsm-handbrake", action="store_true", dest="enable_fsm_handbrake",
                               default=None, help="Engage FSM handbrake, stop all task processing")
    parser.add_argument("--fsm-handbrake-timeout", type=int,
                        help="FSM handbrake will automatically be disabled after timeout")
    fsm_handbrake.add_argument("--disable-fsm-handbrake", action="store_false", dest="enable_fsm_handbrake",
                               default=None, help="Disengage FSM handbrake, resume task processing")

    scenario_fsm_handbrake = parser.add_mutually_exclusive_group()
    scenario_fsm_handbrake.add_argument("--enable-scenario-fsm-handbrake", action="store_true", dest="enable_scenario_fsm_handbrake",
                               default=None, help="Engage Scenario FSM handbrake, stop all task processing")
    parser.add_argument("--scenario-fsm-handbrake-timeout", type=int,
                        help="Scenario FSM handbrake will automatically be disabled after timeout")
    scenario_fsm_handbrake.add_argument("--disable-scenario-fsm-handbrake", action="store_false", dest="enable_scenario_fsm_handbrake",
                               default=None, help="Disengage Scenario FSM handbrake, resume task processing")

    checks_percent_subparsers = register_subparsers(subparsers, "checks_percentage",
                                                    "Checks percentage config override control")
    register_parser(checks_percent_subparsers, "list", _on_list_checks_percentage_overrides,
                    "List current checks percentage overrides")

    register_parser(
        checks_percent_subparsers,
        "get",
        _on_get_checks_percentage_override,
        "Get check percent override").add_argument(
        "check", type=str, help="Check name"
    )

    set_check_percentage_override_parser = register_parser(
        checks_percent_subparsers,
        "set",
        _on_set_checks_percentage_override,
        "Set check percent override")

    set_check_percentage_override_parser.add_argument("check", type=str, help="Check name")
    set_check_percentage_override_parser.add_argument("percent", type=int, help="Percent value")

    register_parser(
        checks_percent_subparsers,
        "remove",
        _on_remove_checks_percentage_override,
        "Remove check percent override").add_argument(
        "check", type=str, help="Check name"
    )


    global_timed_limit_subparser = register_subparsers(
        subparsers, "global_timed_limits_overrides", "Global timed limit override"
    )
    register_parser(global_timed_limit_subparser, "list", _on_list_global_timed_limit,
                    "List current global timed limits override")
    set_global_timed_limit_subparser = register_parser(
        global_timed_limit_subparser,
        "set",
        _on_set_global_timed_limit,
        "Override global timed limit")

    set_global_timed_limit_subparser.add_argument("failure_name", type=str, help="Failure name (e.g. max_rebooted_hosts)")
    set_global_timed_limit_subparser.add_argument("period", type=str, help="Time period (1d, 3h, 15m, ...)")
    set_global_timed_limit_subparser.add_argument("limit", type=int, help="Count of failures in period")
    register_parser(
        global_timed_limit_subparser,
        "remove",
        _on_remove_global_timed_limit,
        "Remove global timed limit override").add_argument(
        "failure_name", type=str, help="Failure name"
    )

    automation_plot_subparser = register_subparsers(subparsers, "automation_plot",
                                                    "Automation plot control")
    checks_subparser = register_subparsers(automation_plot_subparser, "checks", "Control automation plot checks")
    enable_parser = register_parser(checks_subparser,
                                    "enable",
                                    _on_automation_plot_check(enable=True),
                                    "Enable check in automation plot",
                                    with_reason=True)
    disable_parser = register_parser(checks_subparser,
                                     "disable",
                                     _on_automation_plot_check(enable=False),
                                     "Disable check in automation plot", with_reason=True)

    for parser in (enable_parser, disable_parser):
        parser.add_argument("automation_plot_id", type=str, help="Automation plot id")
        parser.add_argument("check_name", type=str, help="Check name")


def _on_list(client, args):
    for name, value in client.get_settings().items():
        if isinstance(value, bool):
            value = str(value).lower()
        print("{name}: {value}".format(name=name, value=value))


def _on_automation(disable):
    def _handler(client, args):
        question_user(args, "Are you sure want to {} all automation types globally?",
                      "disable" if disable else "enable")

        params = {"disable_automation": disable}
        if args.reason is not None:
            params["reason"] = args.reason

        client.set_settings(**params)

    return _handler


def _on_set(client, args):
    params = dict()
    if args.healing is not None:
        params["disable_healing"] = not args.healing

    if args.dns_automation is not None:
        params["disable_dns_automation"] = not args.dns_automation

    if args.inventory_invalid_hosts_limit is not None:
        params["inventory_invalid_hosts_limit"] = args.inventory_invalid_hosts_limit

    if args.enable_fsm_handbrake is not None:
        params["enable_fsm_handbrake"] = args.enable_fsm_handbrake
    
    if args.fsm_handbrake_timeout is not None:
        if args.enable_fsm_handbrake is not None and not args.enable_fsm_handbrake:
            print("argument --fsm-handbrake-timeout: not allowed with argument --disable-fsm-handbrake",
                    file=sys.stderr)
            exit(2)
        params["fsm_handbrake_timeout"] = args.fsm_handbrake_timeout

    if args.enable_scenario_fsm_handbrake is not None:
        params["enable_scenario_fsm_handbrake"] = args.enable_scenario_fsm_handbrake

    if args.scenario_fsm_handbrake_timeout is not None:
        if args.enable_scenario_fsm_handbrake is not None and not args.enable_scenario_fsm_handbrake:
            print("argument --scenario-fsm-handbrake-timeout: not allowed with argument --disable-scenario-fsm-handbrake",
                    file=sys.stderr)
            exit(2)
        params["scenario_fsm_handbrake_timeout"] = args.scenario_fsm_handbrake_timeout

    if not params:
        return

    question_user(args,
                  "Are you sure want to apply these settings?\n{}",
                  "\n".join(map("=".join, map(lambda kv: map(str, kv), params.items()))))

    if args.reason is not None:
        params["reason"] = args.reason

    client.set_settings(**params)


def _on_list_checks_percentage_overrides(client, args):
    overrides = client.get_checks_overrides()['result']

    if not overrides:
        print("no overrides defined")
        return

    for i in overrides:
        print("{}: {}".format(i["check"], i["percent"]))


def _on_get_checks_percentage_override(client, args):
    override = client.get_check_override(args.check)
    print("{}: {}".format(override["check"], override["percent"]))


def _on_set_checks_percentage_override(client, args):
    override = client.set_check_override(args.check, args.percent)
    print("{}: {}".format(override["check"], override["percent"]))


def _on_remove_checks_percentage_override(client, args):
    override = client.remove_check_override(args.check)
    print("removed override: '{}: {}'".format(override["check"], override["percent"]))


def _on_list_global_timed_limit(client, args):
    overrides = client.get_global_timed_limits()

    if not overrides:
        print("no overrides defined")
        return

    for failure_name, timed_limit in overrides.items():
        print("{}: {} per {}".format(failure_name, timed_limit["limit"], timed_limit["period"]))


def _on_set_global_timed_limit(client, args):
    override = client.set_global_timed_limit(args.failure_name, args.period, args.limit)
    print("{}: {} per {}".format(args.failure_name, override["limit"], override["period"]))


def _on_remove_global_timed_limit(client, args):
    override = client.remove_global_timed_limit(args.failure_name)
    print("removed override: '{}: {} per {}'".format(args.failure_name, override["limit"], override["period"]))


def _on_automation_plot_check(enable):
    def _handler(client, args):
        params = {
            "id": args.automation_plot_id,
            "check_name": args.check_name
        }
        if args.reason is not None:
            params["reason"] = args.reason

        if enable:
            method = client.enable_check
            action_str = "enable"
        else:
            method = client.disable_check
            action_str = "disable"

        question_user(args, "Are you sure you want to {} check {} in automation plot {}?",
                      action_str, args.check_name, args.automation_plot_id)
        method(**params)

    return _handler
