"""Contains all logic of processing a task on host group."""

import sys
import time

from walle_api.client import WalleClientError
from walle_api.constants import HostState, HostStatus

from walle_cli.common import Error, LogicalError, unique_list


class TaskType:
    POWER_ON = "power-on"
    POWER_OFF = "power-off"
    REBOOT = "reboot"
    CHECK_DNS = "check-dns"
    PROFILE = "profile"
    REDEPLOY = "redeploy"
    PREPARE = "prepare"
    ALL = [POWER_ON, POWER_OFF, REBOOT, CHECK_DNS, PROFILE, PREPARE, REDEPLOY]


class _TaskProcessor(object):
    def __init__(self, client, group):
        if "task" not in group:
            raise Error("Group '{}' doesn't have any assigned task to process.", group["name"])

        self.__client = client
        self.__group = group
        self.__in_progress_bar = False

        user_order_map = {name: order for order, name in enumerate(group["hosts"])}
        self.__user_order_sort = lambda hosts, reverse=False: \
            sorted(hosts, key=lambda name: user_order_map[name], reverse=reverse)

    def process(self):
        group, task = self.__group, self.__group["task"]

        # Normalize host lists in case of manual editing
        hosts = set(task["hosts"])
        failed = set(task.get("failed", [])) & hosts
        completed = (set(task.get("completed", [])) & hosts) - failed
        in_process = (set(task.get("in_process", [])) & hosts) - completed - failed
        task["hosts"], task["in_process"], task["completed"], task["failed"] = \
            unique_list(task["hosts"]), self.__user_order_sort(in_process), self.__user_order_sort(completed), \
            self.__user_order_sort(failed)

        self.__log_info("Processing '{}' task on group '{}'...", task["type"], group["name"])
        if task["in_process"]:
            self.__log_info("The following hosts are in process: {}.", ", ".join(task["in_process"]))

        try:
            self.__process()
        finally:
            self.__complete_progress_bar()

    def __process(self):
        group = self.__group
        task = group["task"]

        pending = self.__user_order_sort(
            set(task["hosts"]) - set(task["in_process"]) - set(task["completed"]) - set(task["failed"]), reverse=True)

        while True:
            self.__check_in_process()
            if not task["in_process"] and not pending:
                break

            while pending and len(task["in_process"]) < group["process_queue_size"]:
                self.__check_fail_percent()

                name = pending.pop()
                self.__log_info("Processing host {}...", name)
                task["in_process"].append(name)

                allowed_params = ["disable_admin_requests", "reason"]

                if task["type"] != TaskType.POWER_OFF:
                    allowed_params.append("check")

                if task["type"] not in (TaskType.POWER_ON, TaskType.PREPARE, TaskType.CHECK_DNS):
                    allowed_params.append("ignore_cms")

                if task["type"] not in (TaskType.POWER_OFF, TaskType.REDEPLOY, TaskType.PREPARE):
                    allowed_params.append("with_auto_healing")

                if task["type"] == TaskType.POWER_ON:
                    handler = self.__client.power_on_host
                elif task["type"] == TaskType.POWER_OFF:
                    handler = self.__client.power_off_host
                elif task["type"] == TaskType.REBOOT:
                    handler = self.__client.reboot_host
                    allowed_params.extend(("ssh", ))
                elif task["type"] == TaskType.CHECK_DNS:
                    handler = self.__client.check_host_dns
                elif task["type"] == TaskType.PROFILE:
                    handler = self.__client.profile_host
                    allowed_params.extend(("profile", "profile_tags"))
                elif task["type"] == TaskType.REDEPLOY:
                    handler = self.__client.redeploy_host
                    allowed_params.extend(("config", "deploy_tags"))
                elif task["type"] == TaskType.PREPARE:
                    handler = self.__client.prepare_host
                    allowed_params.extend((
                        "profile", "profile_tags", "provisioner", "config", "deploy_tags",
                        "restrictions", "skip_profile"))
                else:
                    raise LogicalError()

                try:
                    handler(name, **{k: v for k, v in task.get("params", {}).items() if k in allowed_params})
                except WalleClientError as e:
                    self.__log_error("Failed to '{}' host {}: {}", task["type"], name, e)
                    self.__fail_host(name)

            group.save(only_if_changed=True)

            self.__draw_progress_bar()
            time.sleep(3)

        self.__log_info("The '{}' task has completed. {} hosts processed, {} failed. Fail percent: {:.1f}",
                        task["type"], len(task["hosts"]), len(task["failed"]), self.__get_fail_percent())
        del group["task"]
        group.save()

    def __check_in_process(self):
        task = self.__group["task"]
        if not task["in_process"]:
            return

        try:
            task_status = {
                TaskType.POWER_ON:  HostStatus.POWERING_ON,
                TaskType.POWER_OFF: HostStatus.POWERING_OFF,
                TaskType.REBOOT:    HostStatus.REBOOTING,
                TaskType.CHECK_DNS: HostStatus.CHECKING_DNS,
                TaskType.PROFILE:   HostStatus.PROFILING,
                TaskType.REDEPLOY:  HostStatus.DEPLOYING,
                TaskType.PREPARE:   HostStatus.PREPARING,
            }[task["type"]]
        except KeyError:
            raise LogicalError()

        try:
            hosts = self.__client.get_hosts(names=task["in_process"])["result"]
        except WalleClientError as e:
            self.__log_error("Failed to get current host status: {}", e)
            return

        for host in hosts:
            name, state, status = host["name"], host["state"], host["status"]
            if name not in task["in_process"]:
                raise Error("Got an unexpected host {} in API response.", name)

            if state == HostState.ASSIGNED and (
                task["type"] != TaskType.POWER_OFF and status == HostStatus.READY or
                task["type"] == TaskType.POWER_OFF and status == HostStatus.MANUAL
            ):
                self.__log_info("Host '{}' has completed.", name)
                task["in_process"].remove(name)
                task["completed"].append(name)
            elif status == HostStatus.DEAD:
                self.__log_error("Host '{}' has failed.", name)
                self.__fail_host(name)
            elif status == task_status:
                pass
            else:
                self.__log_error("Host '{}' got an unexpected status '{}:{}'. Consider it as failed.",
                                 name, state, status)
                self.__fail_host(name)

    def __fail_host(self, name):
        task = self.__group["task"]
        task["in_process"].remove(name)
        task["failed"].append(name)

    def __get_fail_percent(self):
        task = self.__group["task"]
        return float(len(task["failed"])) / len(task["hosts"]) * 100

    def __check_fail_percent(self):
        fail_percent = self.__get_fail_percent()
        if fail_percent >= self.__group["max_fail_percent"]:
            raise Error("Too big task failure percent: {:.1f}. Aborting.", fail_percent)

    def __log_info(self, *args, **kwargs):
        self.__log(sys.stdout, *args, **kwargs)

    def __log_error(self, *args, **kwargs):
        self.__log(sys.stderr, *args, **kwargs)

    def __log(self, *args, **kwargs):
        stream, message, args = args[0], args[1], args[2:]
        if args or kwargs:
            message = message.format(*args, **kwargs)

        self.__complete_progress_bar()
        print(message, file=stream)

    def __draw_progress_bar(self):
        print(".", end="")
        sys.stdout.flush()
        self.__in_progress_bar = True

    def __complete_progress_bar(self):
        if self.__in_progress_bar:
            print()
            self.__in_progress_bar = False


def process_task(client, group):
    _TaskProcessor(client, group).process()
