import stat
import sys

import argparse
import os

import argcomplete
import subprocess

import database.s3_mds.client as s3_mds_client


def check_path_is_file(path):
    assert os.path.isfile(path), f"Path '{path}' is not file"
    assert os.access(path, os.R_OK), f"Path '{path}' is not readable"
    assert os.stat(path)[stat.ST_MODE], f"File '{path}' not executable"


def check_path_is_dir(path):
    assert os.path.isdir(path), f"Path '{path}' is not dir"


def is_tool(name):
    rc = subprocess.call(["which", name], stdout=subprocess.DEVNULL)
    return rc == 0


def execute_command(*args):
    try:
        return subprocess.check_output([str(i) for i in args]).decode('utf-8')
    except subprocess.CalledProcessError as e:
        raise AssertionError('{} returned with error {}'.format(args[0], e.output))


def pretty_output(*args):
    print('||====')
    print('||', *args)


class ArgsProcessor():
    action_name = 'none'  # backup or restore
    description = ""

    common_args = []
    common_kwargs = {}

    prod_args = []
    prod_kwargs = {}

    dev_args = []
    dev_kwargs = {}

    def __init__(self):

        args = self._parse_args()

        self.mode = args.mode

        # Set S3 keys and bucket
        for attr_name, env_key in [("s3_key_id", "WALLE_S3_KEY_ID"), ("s3_secret_key", "WALLE_S3_SECRET_KEY"),
                                   ("s3_bucket", "WALLE_S3_BUCKET")]:
            if '--env-s3-vars' in sys.argv:
                value = os.environ.get(env_key, None)
                assert value, "With flag '--env-s3-vars' S3 auth keys and bucket must be " \
                              "defined in enviroment variables with prefix 'WALLE_*'"
            else:
                value = getattr(args, attr_name)
            setattr(self, attr_name, value)

        # Positional args
        args_to_parse = [self.common_args]
        if self.mode == 'prod':
            args_to_parse.append(self.prod_args)
        else:
            args_to_parse.append(self.dev_args)

        for arg_to_parse in args_to_parse:
            for attr_name, _ in arg_to_parse:
                setattr(self, attr_name[0], getattr(args, attr_name[0], None))

        # Optional args
        kwargs_to_parse = [self.common_kwargs]
        if self.mode == 'prod':
            kwargs_to_parse.append(self.prod_kwargs)
        else:
            kwargs_to_parse.append(self.dev_kwargs)

        for kwarg_to_parse in kwargs_to_parse:
            for dest in kwarg_to_parse:
                setattr(self, dest, getattr(args, dest, None))

        self.s3_client = s3_mds_client.S3Client(self.s3_key_id, self.s3_secret_key, s3_mds_client.S3Client.PROD_URL)

    def db_is_master(self):
        args = [self.mongo_path, "--ipv6", "--host", self.host, "--port", self.port]
        if self.mode == 'prod':
            args.extend(["--username", self.username, "--password", self.password, "--authenticationDatabase",
                         self.authentication_database])
        args.extend(["--quiet", "--eval", "db.isMaster().ismaster"])
        res = execute_command(*args)
        return str(res).strip() == 'true'

    def _parse_args(self):
        parser = argparse.ArgumentParser(description=self.description,
                                         formatter_class=argparse.RawDescriptionHelpFormatter)

        subparsers = parser.add_subparsers(dest="mode", required=True)

        parser_prod = subparsers.add_parser('prod',
                                            help=f'{self.action_name.capitalize()} on production: user, password '
                                                 'and authentication database will be required')
        parser_dev = subparsers.add_parser('dev',
                                           help=f'{self.action_name.capitalize()} on local development server: user, password '
                                                'and authentication database will not be required')

        # If --env-s3-vars is set, we try to get WALLE_S3_KEY_ID, WALLE_S3_SECRET_KEY, WALLE_S3_BUCKET from os.environ
        if '--env-s3-vars' not in sys.argv:
            for parser_ in (parser_prod, parser_dev):
                parser_.add_argument("s3_key_id", type=str, help="s3 key id")
                parser_.add_argument("s3_secret_key", type=str, help="s3 secret key")
                parser_.add_argument("s3_bucket", type=str, help="s3 bucket name")

        # Positional args
        for args, kwargs in self.common_args:
            parser_prod.add_argument(*args, **kwargs)
            parser_dev.add_argument(*args, **kwargs)

        for args, kwargs in self.prod_args:
            parser_prod.add_argument(*args, **kwargs)

        for args, kwargs in self.dev_args:
            parser_dev.add_argument(*args, **kwargs)

        # Optional args
        for dest, (args, kwargs) in self.common_kwargs.items():
            parser_prod.add_argument(*args, dest=dest, **kwargs)
            parser_dev.add_argument(*args, dest=dest, **kwargs)

        for dest, (args, kwargs) in self.prod_kwargs.items():
            parser_prod.add_argument(*args, dest=dest, **kwargs)

        for dest, (args, kwargs) in self.dev_kwargs.items():
            parser_dev.add_argument(*args, dest=dest, **kwargs)

        argcomplete.autocomplete(parser)
        args = parser.parse_args(sys.argv[1:])

        return args

    def main(self):
        raise NotImplementedError()
