import re
import sys
import time
import os
import glob
import random

import requests

import database.s3_mds.client as s3_mds_client
import json

from database.backup_restore_common import ArgsProcessor, execute_command, pretty_output, \
    check_path_is_dir, is_tool, check_path_is_file


class BackupProcessor(ArgsProcessor):
    action_name = 'backup'

    description = """Tool for creating backup
    use 'wall-e-mongodb-backup' as s3_bucket to backup main db
    use 'wall-e-health-backup' as s3_bucket to backup health db
    s3_key_id and s3_secret can be taken from loop.conf file in https://nanny.yandex-team.ru/ui/#/services/catalog/production_wall-e_mongodb/files

    Backup filename example: 'sas2-1200/mongobackup_1565222490.arz'
                             {sas2-1200} - host
                             {mongobackup} - backup_prefix
                             {1565222490} - timestamp
    """

    common_args = [
        (["host"], {"type": str,
                    "help": "Host from which to backup"}),
        (["backup_dir"], {"type": str,
                          "help": "Path to backup"}),
        (["backup_prefix", ], {"type": str,
                               "help": "Backup prefix"}),
        (["rs_name", ], {"type": str,
                         "help": "Replicaset name"}),
        (["backup_keep"], {"type": int,
                           "help": "specify how many backups should rest"}),
    ]

    common_kwargs = {
        "env_s3_vars": (["--env-s3-vars"],
                        {"action": "store_true", "help": "If flag is set S3 args will be taken from env variables:"
                                               "'WALLE_S3_KEY_ID', 'WALLE_S3_SECRET_KEY', 'WALLE_S3_BUCKET'"}),
        "port": (["-P", "--Port"], {"type": int,
                                    "help": "Host port, default: 27017",
                                    "default": 27017}),
        "mongo_dump_path": (["-dp", "--mongodump-path"], {"type": str,
                                                          "help": "mongodump executable path, default: mongodump",
                                                          "default": "mongodump"}),
        "mongo_path": (["-mp", "--mongo-path"], {"type": str,
                                                          "help": "mongo executable path, default: mongo",
                                                          "default": "mongo"}),
        "period": (["--period"], {"type": int,
                                  "help": "Period between backups in seconds",
                                  "default": None})
    }

    prod_args = [
        (["username"], {"type": str,
                        "help": "mongodb user"}),
        (["password"], {"type": str,
                        "help": "mongodb password"}),
        (["authentication_database"], {"type": str, "help": "Auth database"}),
    ]

    dev_kwargs = {
        "username": (["-u", "--username"],
                     {"type": str, "help": "mongodb user"}),
        "password": (["-p", "--password"],
                     {"type": str, "help": "mongodb password"}),
    }

    def backup(self, fetch_path):
        command = [self.mongo_dump_path, "--host", "localhost", "--port", self.port, "--oplog", "--gzip",
                   f"--archive=\"{fetch_path}\""]
        if self.mode == "prod":
            command.extend(["--username", self.username, "--password", self.password, "--authenticationDatabase",
                            self.authentication_database])

        return execute_command(*command)

    def clean_s3(self):
        s3_backup_prefix = f"{self.host}/{self.backup_prefix}"

        req_result = self.s3_client.request("get", self.s3_bucket).content
        backups_list = self.s3_client.return_list(req_result, path="Contents", formatter=s3_mds_client.format_file_item)
        filtered_backups = self.filter_backups(backups_list, s3_backup_prefix)
        to_remove = len(filtered_backups) - self.backup_keep
        if to_remove > 0:
            pretty_output(f"Removing {to_remove} backups from S3:")
            filtered_backups.sort()
            for backup in filtered_backups[:to_remove]:
                print('||  ', backup)
                self.s3_client.request("delete", self.s3_bucket, backup)
        else:
            pretty_output("No backups to remove")

    def clean_local(self):
        filtered_backups = glob.glob(f'{self.backup_dir}/{self.backup_prefix}_*.arz')
        to_remove = len(filtered_backups) - self.backup_keep
        if to_remove > 0:
            pretty_output(f"Removing {to_remove} backups from local:")
            filtered_backups.sort()
            for backup in filtered_backups[:to_remove]:
                try:
                    os.remove(backup)
                    print('||  ', backup)
                except OSError as e:
                    print('|| Unable to remove', backup, ': raise an exception', e)
        else:
            pretty_output("No backups to remove")

    def upload_to_s3(self, filepath):
        filename = os.path.basename(filepath)
        s3_backup_key = f"{self.host}/{filename}"
        data = sys.stdin if filepath is None else open(filepath, 'rb')
        pretty_output("Uploading:", filepath, "to S3", self.s3_bucket, "bucket, with name:", s3_backup_key)
        with data:
            return self.s3_client.request("put", self.s3_bucket, s3_backup_key, data)

    def filter_backups(self, backups_list, backup_prefix=''):
        res = []
        for backup in backups_list:
            match = re.match(rf"^({backup_prefix}.*_([0-9]+).arz)", backup)
            if match:
                res.append(match.group(1))
        return res

    def notify_juggler(self):
        url = "http://localhost:31579/events"
        payload = {
            "events": [
                {
                    "host": self.host,
                    "instance": f"{self.port}",
                    "service": f"mongo-rs-{self.rs_name}-backup",
                    "status": "OK",
                    "tags": [
                        "mongodb",
                        "wall-e.srv",
                        "wall-e-healthdb"
                    ],
                    "description": f"Backup for {self.rs_name} replica set from {self.host}:{self.port} instance completed"
                }
            ]
        }
        headers = {
            'content-type': "application/json"
        }

        response = requests.request("POST", url, data=json.dumps(payload), headers=headers)
        if not response.ok:
            raise Exception("Notify juggler failed with error:", response.reason, "Payload:", payload)

    def main(self):
        self._check_if_mongo_tools_installed()
        check_path_is_dir(self.backup_dir)

        if self.db_is_master():
            print("Host is master, skipping")
            exit()

        fetch_path = f"{self.backup_dir}/{self.backup_prefix}_{int(time.time())}.arz"

        self.backup(fetch_path)
        self.upload_to_s3(fetch_path)

        # remove backups
        self.clean_s3()
        self.clean_local()

        self.notify_juggler()
        time.sleep(self.period + random.uniform(0, self.period / 8))

    def _check_if_mongo_tools_installed(self):
        # If mongo and mongodump exists
        if self.mongo_path == "mongo":
            assert is_tool("mongo"), "Mongo is not installed"
        else:
            check_path_is_file(self.mongo_path)
        if self.mongo_dump_path == "mongodump":
            assert is_tool("mongodump"), "Mongodump is not installed"
        else:
            check_path_is_file(self.mongo_dump_path)


def main():
    backup_processor = BackupProcessor()

    while True:
        backup_processor.main()


if __name__ == '__main__':
    main()
