import re
import sys

import os

import database.s3_mds.client as s3_mds_client

from database.backup_restore_common import ArgsProcessor, execute_command, pretty_output, \
    check_path_is_dir, is_tool, check_path_is_file


class RestoreProcessor(ArgsProcessor):
    action_name = 'restore'
    description = """Tool for restoring mongo backup
    If you are restoring locally, you need to define only s3_key_id, s3_secret_key and s3_bucket
    use 'wall-e-mongodb-backup' as s3_bucket to restore main db
    use 'wall-e-health-backup' as s3_bucket to restore health db
    s3_key_id and s3_secret can be taken from loop.conf file in https://nanny.yandex-team.ru/ui/#/services/catalog/production_wall-e_mongodb/files

    Backup filename example: 'sas2-1200/mongobackup_1565222490.arz'
                             {sas2-1200} - host
                             {mongobackup} - backup_prefix
                             {1565222490} - timestamp
    """

    common_kwargs = {
        "env_s3_vars": (["--env-s3-vars"],
                        {"action": "store_true", "help": "If flag is set S3 args will be taken from env variables:"
                                               "'WALLE_S3_KEY_ID', 'WALLE_S3_SECRET_KEY', 'WALLE_S3_BUCKET'",
                         "required": False}),
        "host": (["-H", "--Host"],
                 {"type": str, "default": "localhost",
                  "help": "Mongo connection host, default is localhost"}),
        "port": (["-P", "--Port"],
                 {"type": int,
                  "help": "Mongo connection port, default value: 27017",
                  "default": 27017}),
        "backup_dir": (["-b", "--backup-dir"],
                       {"type": str,
                        "help": "Path to load local backup if no remote backup found"}),
        "host_source": (["--host-prefix", ],
                          {"type": str,
                           "help": "Backup source host, default: any"}),
        "backup_prefix": (["-r", "--backup-prefix", ],
                          {"type": str,
                           "help": "Backup prefix, default value: 'mongobackup'",
                           "default": "mongobackup"}),
        "dry_run": (["-n", "--dry-run"],
                    {"action": "store_true",
                     "help": "do not actually restore backup, just list all and fetch matching"}),
        "backup_file": (["-f", "--backup-file"],
                        {"type": str,
                         "help": "restore from specified backup file (use dry-run to find all available)"}),
        "nsExclude": (["--nsExclude"], {"type": str, "help": "exclude matching namespaces"}),
        "nsInclude": (["--nsInclude"], {"type": str, "help": "include matching namespaces"}),
        "mongo_restore_path": (["-rp", "--mongorestore-path"], {"type": str,
                                                                "help": "mongorestore executable path: mongorestore",
                                                                "default": "mongorestore"}),
        "mongo_path": (["-mp", "--mongo-path"], {"type": str,
                                                 "help": "mongo executable path, default: mongo",
                                                 "default": "mongo"})
    }

    prod_args = [
        (["username"], {"type": str,
                        "help": "mongodb user"}),
        (["password"], {"type": str,
                        "help": "mongodb password"}),
        (["authentication_database"], {"type": str, "help": "Auth database"}),
    ]

    dev_kwargs = {
        "username": (["-u", "--username"],
                     {"type": str, "help": "mongodb user"}),
        "password": (["-p", "--password"],
                     {"type": str, "help": "mongodb password"}),
    }

    def restore(self, fetched_path):
        pretty_output(f"Restoring {fetched_path}!...")

        command = [self.mongo_restore_path, "--host", self.host, "--port", self.port, "--gzip",
                   f"--archive=\"{fetched_path}\""]
        if self.mode == "prod":
            command.extend(["--username", self.username, "--password", self.password, "--authenticationDatabase",
                            self.authentication_database])

        if self.nsExclude:
            command.append("--nsExclude={}".format(self.nsExclude))

        if self.nsInclude:
            command.append("--nsInclude={}".format(self.nsInclude))

        if not self.nsExclude and not self.nsInclude:
            command.append("--oplogReplay")

        return execute_command(*command)

    def find_last_from_s3(self):
        host_source = self.host_source if self.host_source else r'[\w|_|-.]+'
        s3_backup_prefix = f"{host_source}/{self.backup_prefix}"

        req_result = self.s3_client.request("get", self.s3_bucket).content
        backups_list = self.s3_client.return_list(req_result, path="Contents", formatter=s3_mds_client.format_file_item)

        pretty_output('Backups in S3')
        for backup in backups_list:
            print('||  ', backup)
        return self.find_last(backups_list, s3_backup_prefix)

    def find_last_local(self, dirname):
        backups_list = os.listdir(dirname)
        return self.find_last(backups_list)

    def find_last(self, backups_list, backup_prefix=''):
        last_backup_timestamp = 0
        last_backup = None

        for backup in backups_list:
            match = re.match(rf"^({backup_prefix}.*_([0-9]+).arz)", backup)
            if match:
                timestamp = int(match.group(2))
                if timestamp > last_backup_timestamp:
                    last_backup_timestamp = timestamp
                    last_backup = match.group(1)
        return last_backup

    def fetch_from_s3(self, backup):
        pretty_output(f'Fetching {backup}...')
        backup_file = f"/tmp/{os.path.basename(backup)}"

        if os.path.exists(backup_file):
            return backup_file

        with open(backup_file, 'wb') as out:
            out.write(self.s3_client.request("get", self.s3_bucket, backup).content)
        pretty_output(f'Fetching done!: {backup_file}')

        return backup_file

    def main(self):
        self._check_if_mongo_tools_installed()

        if self.backup_dir:
            check_path_is_dir(self.backup_dir)

        assert self.db_is_master(), "Host is not a master"

        last_backup = self.find_last_from_s3()
        fetched_path = None

        if self.backup_file:
            fetched_path = self.fetch_from_s3(self.backup_file)

        elif last_backup:
            fetched_path = self.fetch_from_s3(last_backup)

        if not (fetched_path and os.path.isfile(fetched_path)):
            if not self.backup_dir:
                pretty_output("No local backup path provided")
                sys.exit()

            pretty_output("Can not fetch backup from s3, try to use local backup")
            last_backup = self.find_last_local(self.backup_dir)
            if not last_backup:
                pretty_output("No local backups")
                sys.exit()

            pretty_output("Found local")
            fetched_path = os.path.join(self.backup_dir, last_backup)

        if not self.dry_run:
            self.restore(fetched_path)

    def _check_if_mongo_tools_installed(self):
        # If mongo, mongorestore exists
        if self.mongo_path == "mongo":
            assert is_tool("mongo"), "Mongo is not installed"
        else:
            check_path_is_file(self.mongo_path)
        if self.mongo_restore_path == "mongorestore":
            assert is_tool("mongorestore"), "Mongorestore is not installed"
        else:
            check_path_is_file(self.mongo_restore_path)


def main():
    restore_processor = RestoreProcessor()
    restore_processor.main()


if __name__ == '__main__':
    main()
