import gevent.monkey
gevent.monkey.patch_all()

import argparse
import logging
import signal
import sys

from apscheduler.schedulers.gevent import GeventScheduler as Scheduler

import gevent
import gevent.signal
from gevent.event import Event
from .rs_monitor import replica_set_monitor
from .util import send_checks, print_checks, MINUTE_SECONDS


def setup_monitors(config, sender):
    return [replica_set_monitor(config, sender=sender)]


def setup_logging():
    console = logging.StreamHandler(sys.stderr)
    formatter = logging.Formatter('%(levelname)s: %(message)s')
    console.setFormatter(formatter)
    # root logger
    logging.getLogger('').addHandler(console)
    logging.getLogger('').setLevel(logging.INFO)


def setup_signals(stop_event):
    def terminate():
        print("terminating")
        stop_event.set()

    gevent.signal.signal(signal.SIGINT, terminate)
    gevent.signal.signal(signal.SIGTERM, terminate)
    gevent.signal.signal(signal.SIGQUIT, terminate)


def parse_args():
    parser = argparse.ArgumentParser(description="Monitoring daemon")
    parser.add_argument("-d", "--demonize", default=False, action="store_true", help="Run in background")
    parser.add_argument("-n", "--dry-run", default=False, action="store_true", help="Don't send events to juggler")

    parser.add_argument("-u", "--username", required=True, help="Mongodb user")
    parser.add_argument("-p", "--password", help="Mongodb password")
    parser.add_argument("--hosts", nargs="+", help="Mongodb hosts")
    parser.add_argument("-rs", "--replica-set", required=True, help="Mongodb replicaset")

    return parser.parse_args()


def start_daemon(args, sender):
    stop_event = Event()
    setup_signals(stop_event)

    scheduler = Scheduler()
    for monitor in setup_monitors(config=args, sender=sender):
        scheduler.add_job(monitor.run, trigger="interval", seconds=5 * MINUTE_SECONDS, name=monitor.name)

    try:
        scheduler.start()
        stop_event.wait()
    finally:
        scheduler.shutdown()


def run_once(args, sender):
    for monitor in setup_monitors(config=args, sender=sender):
        logging.info(monitor.name)
        monitor.run()


def get_sender(dry_run):
    if dry_run:
        return print_checks
    else:
        return send_checks


def main():
    args = parse_args()

    setup_logging()
    sender = get_sender(args.dry_run)

    if args.demonize:
        start_daemon(args, sender)
    else:
        run_once(args, sender)


if __name__ == "__main__":
    main()
