from __future__ import absolute_import

import datetime
import logging
from collections import defaultdict
from functools import partial

import pymongo
import pymongo.errors

from juggler.bundles import Status
from .util import with_timeout, DAY_SECONDS, Monitor, get_environ

STATE_PRIMARY = 1
STATE_SECONDARY = 2
STATE_TIME_TIMEOUT = DAY_SECONDS


try:
    from urllib.parse import quote_plus
except ImportError:
    from urllib import quote_plus


@with_timeout(15)
def get_mongodb_status(connection_uri):
    rs_connection_timeout_ms = 2000

    try:
        client = pymongo.MongoClient(connection_uri, serverSelectionTimeoutMS=rs_connection_timeout_ms)
        return client.admin.command("replSetGetStatus")

    except pymongo.errors.ServerSelectionTimeoutError:
        logging.error("Failed to connect to mongodb, no primary available")
        raise


def check_primary(rs_status):
    for member in rs_status['members']:
        if member['stateStr'] == 'PRIMARY':
            return Status.OK, f"{member['name']} is the master on '{rs_status['set']}' replica set"
        else:
            return Status.CRIT, f"No master on '{rs_status['set']}' replica set"


def check_states(rs_status):
    now = datetime.datetime.now()
    status_reasons = defaultdict(list)

    for member in rs_status['members']:
        state_time = now - member['optimeDate']
        status_reason = f"{member['name']} is {member['stateStr']} for {state_time} on '{rs_status['set']}' replica set"

        if member['state'] > STATE_SECONDARY:
            status = Status.CRIT if state_time.total_seconds() > STATE_TIME_TIMEOUT else Status.WARN
        else:
            status = Status.OK

        status_reasons[status].append(status_reason)

    status = max(status_reasons)
    reason = "; ".join(reason for status, reasons in sorted(status_reasons.items(), reverse=True) for reason in reasons)

    return status, reason


def make_connection_uri(config):
    password = quote_plus(get_environ("mongodb_password"))
    username = quote_plus(config.username)
    hosts = ",".join(config.hosts or ["localhost"])
    replica_set = config.replica_set

    return f"mongodb://{username}:{password}@{hosts}/admin?replicaSet={replica_set}"


def replica_set_monitor(config, sender):
    connection_uri = make_connection_uri(config)

    return Monitor(
        name="Check mongodb replica set status",
        setup=partial(get_mongodb_status, connection_uri),
        sender=sender,
        checks={
            "mongo-rs-production_walle-master": check_primary,
            "mongo-rs-production_walle-states": check_states,
        },
    )
